/*
 * tvflash - update firmware flash memory on Mellanox HCAs
 * Copyright (c) 2004 Topspin Communications.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * $Id: tvflash.c 9509 2006-09-18 01:22:23Z roland $
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <termios.h>
#include <fcntl.h>
#include <pci/pci.h>
#include <sys/utsname.h>
#include <limits.h>
#include <ctype.h>
#include <time.h>
#include <netinet/in.h>

#include "firmware.h"

/* Global variables */
struct tvdevice *tvdevices = NULL, *tvdevices_tail = NULL;
unsigned char force = 0, verbose = 1, use_config_space = 0;
char *argv0;
unsigned short cols;

/* Local prototypes */
static void usage(void);

/* Topspin OID */
static const uint8_t default_oid[] = { 0x00, 0x05, 0xad, 0x00 };

/*
 * Simple utility functions
 */

/* Calculate the CRC16 of an image (list of bytes) */
static uint16_t flash_crc16(uint8_t *image, uint32_t size)
{
	uint32_t crc = 0xffff;
	uint32_t i, j;
	uint32_t word;

	for (j = 0; j < size / 4; j++) {
		word = (image[4 * j] << 24) | (image[4 * j + 1] << 16) |
			(image[4 * j + 2] << 8) | image[4 * j + 3];
		for (i = 0; i < 32; i++) {
			if (crc & 0x8000)
				crc = (((crc << 1) | (word >> 31)) ^ 0x100b) & 0xffff;
			else
				crc = ((crc << 1) | (word >> 31)) & 0xffff;
			word = word << 1;
		}
	}

	for (i = 0; i < 16; i++) {
		if (crc & 0x8000)
			crc = ((crc << 1) ^ 0x100b) & 0xffff;
		else
			crc = (crc << 1) & 0xffff;
	}

	crc = crc ^ 0xffff;

	return (crc & 0xffff);
}

/*
 * We need some LE to host macros and those vary depending on the architecture
 * and there are no standardized ones. So we mix the ntoh* and hton* functions
 * with an unconditional swab* to get the desired behavior. Not the most
 * efficient, but it works and is portable.
 */
static uint16_t le16_to_cpu(uint16_t x)
{
	uint16_t i = htons(x);

	return ((i >> 8) & 0xff) | ((i & 0xff) << 8);
}

#if 0
static uint16_t cpu_to_le16(uint16_t x)
{
	uint16_t i = ntohs(x);

	return ((i >> 8) & 0xff) | ((i & 0xff) << 8);
}
#endif

static uint32_t swab32(uint32_t x)
{
	return ((x >> 24) | ((x >> 8) & 0xff00) | ((x & 0xff00) << 8) | (x << 24));
}

#define be16_to_cpu ntohs
#define cpu_to_be16 htons
#define be32_to_cpu ntohl
#define cpu_to_be32 htonl

static int parse_hex_str(char *arg, unsigned char *hexstr, int octets)
{
	if (strchr(arg, ':') != NULL) {
		char *p;
		int octetspercolon;

		/* Determine the format of the GUID. Is it single or double octet? */
		p = strrchr(arg, ':');
		if (*(p + 1) && *(p + 2) && *(p + 3) && *(p + 4) && !*(p + 5))
			octetspercolon = 2;
		else if (*(p + 1) && *(p + 2) && !*(p + 3))
			octetspercolon = 1;
		else {
			fprintf(stderr, "Unknown hex string format\n");
			return -1;
		}

		/*
		 * We don't require a full GUID here. If the user doesn't
		 * specify a full GUID, we'll put the Topspin OID and 0 pad
		 * the rest and use as much of a GUID the user gives
		 */
		do {
			unsigned char value, hex;

			p = strrchr(arg, ':');
			if (p)
				*(p++) = 0;
			else
				p = arg;

			if (octetspercolon == 2) {
				/* There should be no more characters after this */
				if (*(p + 4) != 0)
					return -1;

				hex = toupper(*(p + 2));
				if (hex >= '0' && hex <= '9')
					value = (hex - '0') << 4;
				else if (hex >= 'A' && hex <= 'F')
					value = (hex - 'A' + 10) << 4;
				else
					return -1;

				hex = toupper(*(p + 3));
				if (hex >= '0' && hex <= '9')
					value |= hex - '0';
				else if (hex >= 'A' && hex <= 'F')
					value |= hex - 'A' + 10;
				else
					return -1;

				hexstr[--octets] = value;
			} else {
				/* There should be no more characters after this */
				if (*(p + 2) != 0)
					return -1;
			}

			hex = toupper(*p);
			if (hex >= '0' && hex <= '9')
				value = (hex - '0') << 4;
			else if (hex >= 'A' && hex <= 'F')
				value = (hex - 'A' + 10) << 4;
			else
				return -1;

			hex = toupper(*(p + 1));
			if (hex >= '0' && hex <= '9')
				value |= hex - '0';
			else if (hex >= 'A' && hex <= 'F')
				value |= hex - 'A' + 10;
			else
				return -1;

			hexstr[--octets] = value;
		} while (p > arg && octets > 0);

		/* Data left over */
		if (p > arg)
			return 1;
	} else {
		unsigned long low_guid = strtol(arg, NULL, 16);

		/* FIXME: Check the result of strtol, or do something more intelligent */

		hexstr[octets - 4] = (low_guid >> 24) & 0xff;
		hexstr[octets - 3] = (low_guid >> 16) & 0xff;
		hexstr[octets - 2] = (low_guid >> 8) & 0xff;
		hexstr[octets - 1] = low_guid & 0xff;

		octets -= 4;
	}

	return octets;
}

static int parse_guid(char *arg, unsigned char *new_guid)
{
	int octets;

	octets = parse_hex_str(arg, new_guid, GUID_LEN);

	if (octets < 0)
		return 1;

	if (octets > 0 && octets < 3) {
		fprintf(stderr, "GUID is not 8 octets, but longer than 4 maximum to automatically generate\n");
		return 1;
	}

	if (octets >= 3) {
		memcpy(new_guid, default_oid, 4);

		if (octets > 3)
			/* Zero out the non specified portions */
			memset(new_guid + 3, 0, octets - 3);
	}

	return 0;
}

static int parse_change_options(struct topspin_vsd *vsd, char *cmd)
{
	char *p = cmd, *end = strchr(cmd, 0);

	while (p && p < end) {
		char *pc, *pe;

		pc = strchr(p, ',');
		if (pc) {
			*pc = 0;
			pc++;
		}

		pe = strchr(p, '=');
		if (!pe) {
			fprintf(stderr, "Missing = in token '%s'\n", p);
			return 1;
		}

		*pe = 0;
		pe++;

		if (strcasecmp(p, "auto_upgrade") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_AUTOUPGRADE);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_AUTOUPGRADE);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_enable_port_1") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_1);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_1);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_enable_port_2") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_2);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_2);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_service_scan") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_ENABLE_SCAN);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_ENABLE_SCAN);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_wait_on_error") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_WAIT_ON_ERROR);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_WAIT_ON_ERROR);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_try_forever") == 0) {
			if (strcasecmp(pe, "yes") == 0 ||
			    strcasecmp(pe, "on") == 0 ||
			    strcasecmp(pe, "1") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TRY_FOREVER);
			else if (strcasecmp(pe, "no") == 0 ||
				 strcasecmp(pe, "off") == 0 ||
				 strcasecmp(pe, "0") == 0)
				vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_TRY_FOREVER);
			else {
				fprintf(stderr, "Unknown boolean value '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_type") == 0) {
			vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_TYPE);
			if (strcasecmp(pe, "well_known") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TYPE_WELL_KNOWN);
			else if (strcasecmp(pe, "saved") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TYPE_SAVED);
			else if (strcasecmp(pe, "pxe") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TYPE_PXE);
			else if (strcasecmp(pe, "disable") == 0)
				vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_TYPE_DISABLE);
			else {
				fprintf(stderr, "Unknown boot type '%s'\n", pe);
				return 1;
			}
		} else if (strcasecmp(p, "boot_saved_port") == 0) {
			unsigned long port;

			port = strtoul(pe, NULL, 10);
			if (port > 2) {
				fprintf(stderr, "Invalid port number '%s'\n", pe);
				return 1;
			}

			vsd->boot_port = port;
		} else if (strcasecmp(p, "boot_saved_ioc_num") == 0) {
			unsigned long ioc;

			ioc = strtoul(pe, NULL, 10);
			if (ioc == ULONG_MAX) {
				fprintf(stderr, "Invalid ioc number '%s'\n", pe);
				return 1;
			}

			vsd->boot_ioc_num = ioc;
		} else if (strcasecmp(p, "boot_saved_dgid") == 0) {
			if (parse_hex_str(pe, vsd->boot_dgid, 16) < 0) {
				fprintf(stderr, "Couldn't parse dgid\n");
				return 1;
			}
		} else if (strcasecmp(p, "boot_saved_service_name") == 0) {
			if (parse_hex_str(pe, vsd->boot_service_name, 8) < 0) {
				fprintf(stderr, "Couldn't parse service name\n");
				return 1;
			}
		} else if (strcasecmp(p, "boot_pxe_secs") == 0) {
			unsigned long secs;

			secs = strtoul(pe, NULL, 10);
			if (secs > 250) {
				fprintf(stderr, "Invalid number of seconds '%s'\n", pe);
				return 1;
			}

			vsd->boot_pxe_secs = secs;
		} else {
			fprintf(stderr, "Unknown option '%s'\n", p);
			return 1;
		}

		p = pc;
	}

	return 0;
}

static void inc_guid(unsigned char *guid, unsigned char *new_guid)
{
	int i, carry = 1;

	/* FIXME: Do we really want to try to increment the vendor oid too? */
	for (i = 7; i >= 0; i--) {
		if (carry && guid[i] == 0xFF) {
			new_guid[i] = 0;
			carry = 1;
		} else {
			new_guid[i] = guid[i] + carry;
			carry = 0;
		}
	}
}

unsigned int status_curpos, status_relpos, status_maxpos;
unsigned int status_curbar;
unsigned char status_prev;

static void status_start(unsigned int max)
{
	status_curpos = 0;
	status_relpos = 0;
	status_maxpos = max;

	status_curbar = 0;
	status_prev = 0;

	if (verbose)
		printf("Flashing - ");
}

static void status_update(unsigned char status, unsigned int curpos)
{
	unsigned int bar;

	if (!verbose)
		return;

	curpos += status_relpos;

	bar = (curpos * (cols - 12)) / status_maxpos;
	status_curpos = curpos;

	/* Force a character to be printed during a status change */
	if (status != status_prev && status_curbar >= bar)
		bar = status_curbar + 1;

	if (status_curbar < bar) {
		for (; status_curbar < bar; status_curbar++)
			printf("%c", status);

		fflush(stdout);
	}

	status_prev = status;
}

static void status_mark(void)
{
	status_relpos = status_curpos;
}

static void status_stop(void)
{
	if (verbose)
		printf("\n");
}

/*
 * HCA card management
 */

static int vpd_read(struct tvdevice *tvdev);

static int scan_devices(struct pci_access *pacc)
{
	struct pci_dev *pdev;
	struct tvdevice *tvdev;
	int pci_count = 0, pciconf_count = 0;
	unsigned short kernel;
	struct utsname utsname;

	/*
	 * libpci, depending on the kernel it's running under, will create the
	 * PCI device list in the correct order, or backwards. As a result, we
	 * need to create our list appropriately too so the order of the HCAs
	 * matches what the kernel sees the HCA list (ie ib0 is HCA #0, etc)
	 */
	if (uname(&utsname) < 0) {
		fprintf(stderr, "couldn't get uname information, assuming 2.4 kernel\n");
		kernel = 0x0204;
	} else {
		unsigned int major, minor;

		sscanf(utsname.release, "%d.%d", &major, &minor);
		kernel = ((major & 0xff) << 8) | (minor & 0xff);
	}

	/*
	 * Example of how various configurations appear on the PCI bus.
	 * A1 is first HCA, A0 is second HCA.
	 *
	 * Example of both A1 and A0 HCAs in normal mode:
	 *
	 * 07:00:0 5a44 00a0 (MT23108)
	 * 04:00:0 5a44 00a1 (MT23108)
	 * 06:01:0 5a46 00a0 (PCI bridge)
	 * 03:01:0 5a46 00a1 (PCI bridge)
	 *
	 * Example of A1 HCA in normal mode, A0 HCA in recovery mode:
	 *
	 * 06:01:0 5a45 00a0 (PCI conf)
	 * 04:00:0 5a44 00a1 (MT23108)
	 * 03:01:0 5a46 00a1 (PCI bridge)
	 *
	 * Example of A1 HCA in recovery mode, A0 HCA in normal mode:
	 *
	 * 06:00:0 5a44 00a0 (MT23108)
	 * 05:01:0 5a46 00a0 (PCI bridge)
	 * 03:01:0 5a45 00a1 (PCI conf)
	 */
	for (pdev = pacc->devices; pdev; pdev = pdev->next) {
		int i;

		pci_fill_info(pdev, PCI_FILL_IDENT);

		for (i = 0; i < sizeof(pciids) / sizeof(pciids[0]); i++) {
			if (pciids[i].vendor != pdev->vendor_id ||
					pciids[i].device != pdev->device_id)
				continue;

			tvdev = malloc(sizeof(*tvdev));
			if (!tvdev) {
				fprintf(stderr, "couldn't allocate %d bytes for device struct\n", (int) sizeof *tvdev);
				return -1;
			}
			memset(tvdev, 0, sizeof(*tvdev));
			tvdev->pdev = pdev;
			tvdev->pciid = &pciids[i];
			tvdev->flags = pciids[i].flags;

			vpd_read(tvdev);

			tvdev->revision = pci_read_word(pdev, PCI_CLASS_REVISION) & 0xff;

			/* According to the documentation, a revision of 0x00 is an A0 */
			if (tvdev->revision == 0x00)
				tvdev->revision = 0xA0;
			else if (tvdev->revision == 0x01)
				tvdev->revision = 0xA1;

			if (kernel <= 0x0204) {
				tvdev->next = tvdevices;
				if (!tvdevices_tail)
					tvdevices_tail = tvdev;
				tvdevices = tvdev;
			} else {
				if (tvdevices_tail)
					tvdevices_tail->next = tvdev;
				else /* first device we found */
					tvdevices = tvdev;
				tvdevices_tail = tvdev;
			}
		}
	}

	for (tvdev = tvdevices; tvdev; tvdev = tvdev->next) {
		if (tvdev->flags & FLAG_RECOVERY)
			tvdev->num = pciconf_count++;
		else
			tvdev->num = pci_count++;
	}

	return pci_count + pciconf_count;
}

static struct tvdevice *find_device(int hca)
{
	struct tvdevice *tvdev;

	for (tvdev = tvdevices; tvdev && hca; tvdev = tvdev->next, hca--)
		;

	return tvdev;
}

/*
 * Low level flash functions (flash device, GPIO)
 */

/*
 * VPD code
 */

static int read_vpd_data(struct pci_dev *pdev, unsigned int vpd_off,
	unsigned int addr, unsigned int *data)
{
	int i;
	unsigned int val;
	struct timeval tstart, tcur;
	unsigned long elapsed;

	gettimeofday(&tstart, NULL);

	i = pci_write_long(pdev, vpd_off, addr << 16);
	do {
		val = pci_read_long(pdev, vpd_off);
		if ((val >> 31) & 0x01) {
			val = htonl(pci_read_long(pdev, vpd_off + 4));
			*data = val;
			return 0;
		}
		gettimeofday(&tcur, NULL);
		elapsed = ((tcur.tv_sec - tstart.tv_sec) * 1000L) +
				((tcur.tv_usec - tstart.tv_usec) / 1000L);
	} while (elapsed < 100);

	if (verbose > 1)
		fprintf(stderr, "read_vpd_data: timed out at off %x\n", addr);

	return 1;
}

static int vpd_exists(struct pci_dev *pdev, int *vpd_off)
{
	int cap_ptr, cap_id;
	int val;

	cap_ptr = pci_read_long(pdev, PCI_CAPABILITY_LIST) & 0xff;
	while (cap_ptr != 0) {
		val = pci_read_long(pdev, cap_ptr);
		cap_id = val & 0xff;
		if (cap_id == PCI_CAP_ID_VPD) {
			*vpd_off = cap_ptr;
			return 1;
		}
		cap_ptr = (val >> 8) & 0xff;
	}

	return 0;
}

static int vpd_read(struct tvdevice *tvdev)
{
	int vpd_off;

	tvdev->vpd_present = 0;

	if (vpd_exists(tvdev->pdev, &vpd_off)) {
		unsigned int *ptr;
		int i, swap_endian = 0;

		ptr = tvdev->vpd.vpd_int;

		if (read_vpd_data(tvdev->pdev, vpd_off, 0, ptr++) != 0)
			return 0;

		/* Check to see if we got the endianess swapped on us */
		if (tvdev->vpd.vpd_char[0] != 0x82 && tvdev->vpd.vpd_char[3] == 0x82)
			swap_endian = 1;

		for (i = 1; i < 64; i++) {
			if (read_vpd_data(tvdev->pdev, vpd_off, (i * 4), ptr++) != 0) {
				if (i < 32)
					return 0;
				else
					break;
			}
		}

		tvdev->vpd_present = 1;

		if (swap_endian) {
			ptr = tvdev->vpd.vpd_int;
			for (i = 0; i < 64; i++) {
				*ptr = swab32(*ptr);
				ptr++;
			}
		}
	}

	return tvdev->vpd_present;
}

/*
 * Read/Write from the configuration space
 */
static unsigned int read_cfg(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned int val;

	switch (tvdev->method) {
	case METHOD_MMAP:
		val = ntohl(*(uint32_t *)(tvdev->bar0 + addr));
		break;
	case METHOD_PCI_CFG:
		if (!pci_write_long(tvdev->pdev, 22 * 4, addr)) {
			fprintf(stderr, "read_cfg: pci_write_long failed\n");
			exit(1);
		}

		val = pci_read_long(tvdev->pdev, 23 * 4);
		break;
	default:
		abort();
	}

	return val;
}

static void write_cfg(struct tvdevice *tvdev, unsigned int addr,
	unsigned int data)
{
	switch (tvdev->method) {
	case METHOD_MMAP:
		*(uint32_t *)(tvdev->bar0 + addr) = htonl(data);
		break;
	case METHOD_PCI_CFG:
		if (!pci_write_long(tvdev->pdev, 22 * 4, addr) ||
				!pci_write_long(tvdev->pdev, 23 * 4, data)) {
			fprintf(stderr, "write_cfg: pci_write_long failed\n");
			exit(1);
		}
		break;
	default:
		abort();
	}
}

#define NUM_SPIS	4

#define FLASH_GW	 0xF0400
#define FLASH_ADDR 0xF0404
#define FLASH_DATA 0xF0408
#define FLASH_CS	 0xF0418

#define READ_OP		 (1 << 0)

#define SPI_NO_DATA (1 << 4)
#define SPI_NO_ADDR (1 << 5)
#define SPI_SPECIAL (1 << 6)
#define BUSY				(1 << 30)

#define WIP				 (1 << 24)
#define WEL				 (1 << 25)

#define FC_SE		0xd8
#define FC_PP		0x02
#define FC_RDSR		0x05
#define FC_RDID		0x9f
#define FC_WREN		0x06

#define SPI_RDID_M25P80 0xffffffff
#define SPI_RDID_M25P16 0x20201500

#define WAIT_INTERVAL 10

/* Discover the devices connected to the Sinai SPI interface */
void spi_discover_serial_proms(struct tvdevice *tvdev)
{
	unsigned int status, sp_num, rdid_val;

	tvdev->flash_num_spis = 0;

	do {
		status = read_cfg(tvdev, FLASH_GW);
	} while (status & BUSY);

	for (sp_num = 0; sp_num < NUM_SPIS; sp_num++) {
		write_cfg(tvdev, FLASH_CS, sp_num << 30);
		write_cfg(tvdev, FLASH_ADDR, FC_RDSR << 24);
		write_cfg(tvdev, FLASH_GW, READ_OP | SPI_NO_ADDR | SPI_SPECIAL | BUSY);

		do {
			status = read_cfg(tvdev, FLASH_GW);
		} while (status & BUSY);

		if ((read_cfg(tvdev, FLASH_DATA) >> 24) == 0)
			tvdev->flash_num_spis++;
	}

	/* Discover the type of the first SPI device - it's assumed they are all the same */
	write_cfg(tvdev, FLASH_CS, 0 << 30);
	write_cfg(tvdev, FLASH_ADDR, FC_RDID << 24);
	write_cfg(tvdev, FLASH_GW, READ_OP | SPI_NO_ADDR | SPI_SPECIAL | BUSY | (2 << 8));

	do {
		status = read_cfg(tvdev, FLASH_GW);
	} while (status & BUSY);

	rdid_val = read_cfg(tvdev, FLASH_DATA);

	switch(rdid_val) {
	case SPI_RDID_M25P80:
		tvdev->flash_spi_sp_type = SP_ST_M25P80;
		tvdev->flash_size = tvdev->flash_num_spis * 0x100000;
		tvdev->flash_bank_shift = 20;
		break;

	case SPI_RDID_M25P16:
		tvdev->flash_spi_sp_type = SP_ST_M25P16;
		tvdev->flash_size = tvdev->flash_num_spis * 0x200000;
		tvdev->flash_bank_shift = 21;
		break;

	default:
		tvdev->flash_spi_sp_type = SP_UNKNOWN;
		/* Assume some conservative settings */
		tvdev->flash_num_spis = 1;
		tvdev->flash_size = 0x100000;
		tvdev->flash_bank_shift = 20;
	}

	write_cfg(tvdev, FLASH_CS, 0 << 30);
	tvdev->flash_bank = 0;

	tvdev->flash_command_set = CS_SPI;
	tvdev->flash_sector_sz = 0x10000;	/* 64KB */
}

static void flash_set_bank(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned char bank = addr >> tvdev->flash_bank_shift;

	if (bank == tvdev->flash_bank)
		return;

	switch (tvdev->flash_command_set) {
	case CS_SPI:
		write_cfg(tvdev, FLASH_CS, bank << 30);
		break;
	default:
		/* Set the appropriate data bits */
		write_cfg(tvdev, GPIO_DAT + 4, (read_cfg(tvdev, GPIO_DAT + 4) & ~(0x07 << 4)) | (bank << 4));
		break;
	}

	tvdev->flash_bank = bank;
}

/* read a dword from the flash, address must be 4 bytes aligned */
static unsigned int flash_read(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned int cmd, status;

	flash_set_bank(tvdev, addr);

	switch (tvdev->flash_command_set) {
	case CS_SPI:
		write_cfg(tvdev, FLASH_ADDR, (addr & tvdev->flash_bank_mask));
		write_cfg(tvdev, FLASH_GW, READ_OP | BUSY | (2 << 8));

		do {
			status = read_cfg(tvdev, FLASH_GW);
		} while (status & BUSY);

		return read_cfg(tvdev, FLASH_DATA);
	default:
		write_cfg(tvdev, 0xF01A4, (addr & (tvdev->flash_bank_mask & ~3)) | (1 << 29));
		do {
			cmd = read_cfg(tvdev, 0xF01A4);
		} while (cmd & 0xE0000000);

		return read_cfg(tvdev, 0xF01A8);
	}
}

/* read a byte from the flash */
static unsigned char flash_byte_read(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned int data;

	data = flash_read(tvdev, addr & ~3);
	return ((data >> ((3 - (addr & 3)) * 8)) & 0xFF);
}

/* writes a command to the flash */
static void flash_write_cmd(struct tvdevice *tvdev, unsigned int addr,
	unsigned char data)
{
	unsigned int cmd;

	write_cfg(tvdev, 0xF01A8, data << 24);
	write_cfg(tvdev, 0xF01A4, (addr & tvdev->flash_bank_mask) | (2 << 29));
	do {
		cmd = read_cfg(tvdev, 0xF01A4);
	} while (cmd & 0xE0000000);
}

static void flash_chip_reset(struct tvdevice *tvdev)
{
	/* Issue Flash Reset Command */
	switch (tvdev->flash_command_set) {
	case CS_INTEL:
		flash_write_cmd(tvdev, 0x555, 0xFF);
		break;
	case CS_AMD:
		flash_write_cmd(tvdev, 0x555, 0xF0);
		break;
	default:
		break;
	}
}

static void spi_wait_wip(struct tvdevice *tvdev)
{
	unsigned int status;
	unsigned int count = 0;

	do {
		status = read_cfg(tvdev, FLASH_GW);
	} while (status & BUSY);

	do {
		if (++count > 5000)
			usleep(WAIT_INTERVAL);

		write_cfg(tvdev, FLASH_ADDR, FC_RDSR << 24);
		write_cfg(tvdev, FLASH_GW, READ_OP | SPI_NO_ADDR | SPI_SPECIAL | BUSY);

		do {
			status = read_cfg(tvdev, FLASH_GW);
		} while (status & BUSY);

		status = read_cfg(tvdev, FLASH_DATA);
	} while (status & WIP);
}

static void flash_erase_sector(struct tvdevice *tvdev, unsigned int addr)
{
	unsigned int status;

	flash_set_bank(tvdev, addr);

	/* Issue Flash Sector Erase Command */
	switch (tvdev->flash_command_set) {
	case CS_SPI:
		/* Issue Write Enable */
		write_cfg(tvdev, FLASH_ADDR, FC_WREN << 24);
		write_cfg(tvdev, FLASH_GW, SPI_NO_ADDR | SPI_NO_DATA | SPI_SPECIAL | BUSY);

		do {
			status = read_cfg(tvdev, FLASH_GW);
		} while (status & BUSY);

		/* Issue Sector Erase */
		write_cfg(tvdev, FLASH_ADDR, (FC_SE << 24) | (addr & tvdev->flash_bank_mask));
		write_cfg(tvdev, FLASH_GW, SPI_NO_DATA | SPI_SPECIAL | BUSY);

		/* Wait for sector erase completion */
		spi_wait_wip(tvdev);
		break;
	case CS_INTEL:
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, 0x20); /* Erase */
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, 0xD0); /* Confirm */

		/* Wait for sector erase completion */
		do {
			usleep(WAIT_INTERVAL);
			status = flash_read(tvdev, addr);
		} while (!(status & 0x80));
		break;
	case CS_AMD:
		flash_write_cmd(tvdev, 0x555, 0xAA);
		flash_write_cmd(tvdev, 0x2AA, 0x55);
		flash_write_cmd(tvdev, 0x555, 0x80);
		flash_write_cmd(tvdev, 0x555, 0xAA);
		flash_write_cmd(tvdev, 0x2AA, 0x55);
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, 0x30);

		/* Wait for sector erase completion */
		do {
			usleep(WAIT_INTERVAL);
			status = flash_read(tvdev, addr);
		} while (status != 0xFFFFFFFF);
		break;
	default:
		break;
	}

	flash_chip_reset(tvdev);
}

static int log2up(unsigned long in)
{
	unsigned int i;

	for (i = 0; i < 32; i++) {
		if (in <= (unsigned long)(1 << i))
			break;
	}

	return i;
}

static void flash_spi_write_block(struct tvdevice *tvdev, unsigned int addr,
	unsigned char *sector, unsigned int len)
{
	unsigned int i, status;

	/* Enable write */
	write_cfg(tvdev, FLASH_ADDR, FC_WREN << 24);
	write_cfg(tvdev, FLASH_GW, SPI_NO_ADDR | SPI_NO_DATA | SPI_SPECIAL | BUSY);

	do {
		status = read_cfg(tvdev, FLASH_GW);
	} while (status & BUSY);

	write_cfg(tvdev, FLASH_ADDR, (FC_PP << 24) | (addr & tvdev->flash_bank_mask));

	for (i = 0; i < len; i += 4) {
		uint32_t dword = *((uint32_t *)(sector + i));

		write_cfg(tvdev, FLASH_DATA + i, be32_to_cpu(dword));
	}

	write_cfg(tvdev, FLASH_GW, SPI_SPECIAL | BUSY | (log2up(len) << 8));

	/* Wait for erase to complete */
	spi_wait_wip(tvdev);
}

static void flash_write_byte(struct tvdevice *tvdev, unsigned int addr,
	unsigned char data)
{
	unsigned int status;

	switch (tvdev->flash_command_set) {
	case CS_INTEL:
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, 0x40);
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, data);

		/* Wait for the Byte Program to Complete (Verify Byte Program) */
		do {
			status = flash_read(tvdev, addr & ~3);
		} while (!(status & 0x80));
		break;
	case CS_AMD:
		/* Issue Byte Program Command */
		flash_write_cmd(tvdev, 0x555, 0xAA);
		flash_write_cmd(tvdev, 0x2AA, 0x55);
		flash_write_cmd(tvdev, 0x555, 0xA0);
		flash_write_cmd(tvdev, addr & tvdev->flash_bank_mask, data);

		/* Wait for the Byte Program to Complete (Verify Byte Program) */
		do {
			status = flash_read(tvdev, addr & ~3);
		} while (data != ((status >> ((3 - (addr & 3)) * 8)) & 0xFF));
		break;
	default:
		/* Not possible, but compiler complains */
		break;
	}
}

static void grab_gpio(struct tvdevice *tvdev)
{
	/* Grab the semaphore first */
	while (read_cfg(tvdev, 0xF03FC) == 1)
		;

	write_cfg(tvdev, 0xF03FC, 1);

	/* Now copy out the values */
	tvdev->gpio_data[0] = read_cfg(tvdev, GPIO_DAT) & 0xFF;
	tvdev->gpio_data[1] = read_cfg(tvdev, GPIO_DAT + 4);

	tvdev->gpio_direction[0] = read_cfg(tvdev, GPIO_DIR) & 0xFF;
	tvdev->gpio_direction[1] = read_cfg(tvdev, GPIO_DIR + 4);

	tvdev->gpio_polarity[0] = read_cfg(tvdev, GPIO_POL) & 0xFF;
	tvdev->gpio_polarity[1] = read_cfg(tvdev, GPIO_POL + 4);

	tvdev->gpio_output_mode[0] = read_cfg(tvdev, GPIO_MOD) & 0xFF;
	tvdev->gpio_output_mode[1] = read_cfg(tvdev, GPIO_MOD + 4);
}

static void release_gpio(struct tvdevice *tvdev)
{
	if (tvdev->board->chip == CHIP_SINAI) {
		flash_set_bank(tvdev, 0);

		/* Unlock GPIO */
		write_cfg(tvdev, GPIO_LOCK, 0xAAAA);
	}

	/* Write back the saved values */
	write_cfg(tvdev, GPIO_DAT, (read_cfg(tvdev, GPIO_DAT) & ~0xFF) | tvdev->gpio_data[0]);
	write_cfg(tvdev, GPIO_DAT + 4, tvdev->gpio_data[1]);

	write_cfg(tvdev, GPIO_DIR, (read_cfg(tvdev, GPIO_DIR) & ~0xFF) | tvdev->gpio_direction[0]);
	write_cfg(tvdev, GPIO_DIR + 4, tvdev->gpio_direction[1]);

	write_cfg(tvdev, GPIO_POL, (read_cfg(tvdev, GPIO_POL) & ~0xFF) | tvdev->gpio_polarity[0]);
	write_cfg(tvdev, GPIO_POL + 4, tvdev->gpio_polarity[1]);

	write_cfg(tvdev, GPIO_MOD, (read_cfg(tvdev, GPIO_MOD) & ~0xFF) | tvdev->gpio_output_mode[0]);
	write_cfg(tvdev, GPIO_MOD + 4, tvdev->gpio_output_mode[1]);

	/* Release the semaphore */
	write_cfg(tvdev, 0xF03FC, 0);
}

static struct board *vpd_identify(struct tvdevice *tvdev)
{
	char str[MAX_STR];
	int i;

	if (tvdev->vpd_present) {
		if (tvdev->vpd.vpd_char[0] == 0x82) {
			unsigned short len16;
			char *p = (char *)&tvdev->vpd.vpd_char[3];

			memcpy(&len16, tvdev->vpd.vpd_char + 1, sizeof(len16));
			len16 = le16_to_cpu(len16);

			/* Skip leading whitespace */
			for (; len16 > 0 && isspace(*p); len16--, p++)
				;

			/* Make sure we don't copy too much */
			if (len16 > sizeof(str))
				len16 = sizeof(str);

			memcpy(str, p, len16);
			str[len16] = 0;

			/* Strip off any trailing whitespace */
			for (len16--; len16 > 0 && isspace(str[len16]); len16--)
				str[len16] = 0;
		} else
			fprintf(stderr, " \nError. String Tag not present (found tag %02x instead)\n",
				tvdev->vpd.vpd_char[0]);
	} else
		str[0] = 0;

	for (i = 0; i < sizeof(board_vpds) / sizeof(board_vpds[0]); i++) {
		if (strcasecmp(str, board_vpds[i].str) == 0)
			return board_vpds[i].board;
	}

	return NULL;
}

static struct board *arbel_identify(struct tvdevice *tvdev)
{
	struct board *board;

	board = vpd_identify(tvdev);
	if (board)
		return board;

	return &genarbel;
}

static struct board *sinai_identify(struct tvdevice *tvdev)
{
	struct board *board;

	board = vpd_identify(tvdev);
	if (board)
		return board;

	return &gensinai;
}

static struct board *tavor_identify(struct tvdevice *tvdev)
{
	struct board *board;

	/* Detect the difference between a Jaguar and Cougar via the GPIO pins */

	/* Set GPIO pin 12 to input and check the value */
	write_cfg(tvdev, GPIO_DIR + 4, read_cfg(tvdev, GPIO_DIR + 4) & ~(1 << 12));

	usleep(10);

	/* Check if board is a Jaguar by looking at GPIO pin 12 */
	if (!(read_cfg(tvdev, GPIO_DAT + 4) & (1 << 12)))
		return &jaguar;

	/* Check the rest by VPD */
	board = vpd_identify(tvdev);
	if (board)
		return board;

	/*
	 * FIXME: We shouldn't assume Cougar if the VPD is corrupt. The very
	 * fact that there is a VPD probably means it's not a Cougar board.
	 * It's only a Cougar if there is no VPD at all.
	 */

	return &cougar;
}

static void fixup_topspin_vsd(struct topspin_vsd *vsd)
{
	if (!(vsd->flags & cpu_to_be32(VSD_FLAG_ALIGN_FIXED))) {
		/*
		 * Some images have the hw_label field misaligned by one byte.
		 * If this is an old image (the flag isn't set), then move
		 * it back one character
		 */
		memmove(vsd->hw_label,
			vsd->hw_label + 1,
			sizeof(vsd->hw_label) - 1);
		vsd->flags |= cpu_to_be32(VSD_FLAG_ALIGN_FIXED);
	}

	if (!(vsd->flags & cpu_to_be32(VSD_FLAG_NEW_BUILD_NUM))) {
		/*
		 * build_num used to be an 8 bit field, but our build numbers
		 * quickly outgrew that limitation, so expand to 16 bits and
		 * a new location
		 */
		vsd->build_num = vsd->old_build_num;
		vsd->flags |= cpu_to_be32(VSD_FLAG_NEW_BUILD_NUM);
	}

	if ((vsd->flags & cpu_to_be32(VSD_FLAG_BOOT_OPTIONS)) &&
			((vsd->boot_version < 1) || (vsd->boot_version > BOOT_VERSION))) {
		/* Invalid, reset to defaults */
		vsd->boot_version = BOOT_VERSION;
		vsd->boot_port = 0;
		vsd->boot_ioc_num = 0;
		memset(vsd->boot_dgid, 0, sizeof(vsd->boot_dgid));
		memset(vsd->boot_service_name, 0, sizeof(vsd->boot_service_name));
		vsd->boot_pxe_secs = BOOT_PXE_SECS_DEFAULT;

		vsd->flags &= ~cpu_to_be32(VSD_FLAG_BOOT_OPTIONS_MASK);
		vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_1 |
					VSD_FLAG_BOOT_ENABLE_PORT_2 |
					VSD_FLAG_BOOT_ENABLE_SCAN |
					VSD_FLAG_BOOT_TYPE_WELL_KNOWN |
					VSD_FLAG_BOOT_GID_BIG);
	}

	if (!(vsd->flags & cpu_to_be32(VSD_FLAG_BOOT_GID_BIG))) {
		uint8_t dword[4];
		int i;

		for (i = 0; i < sizeof(vsd->boot_dgid); i += 4) {
			memcpy(dword, &vsd->boot_dgid[i], 4);
			vsd->boot_dgid[i + 0] = dword[3];
			vsd->boot_dgid[i + 1] = dword[2];
			vsd->boot_dgid[i + 2] = dword[1];
			vsd->boot_dgid[i + 3] = dword[0];
		}

		vsd->flags |= cpu_to_be32(VSD_FLAG_BOOT_GID_BIG);
	}

	if (vsd->boot_version < 2) {
		uint8_t dword[4];
		int i;

		vsd->boot_version = BOOT_VERSION;
		vsd->boot_pxe_secs = BOOT_PXE_SECS_DEFAULT;

		for (i = 0; i < sizeof(vsd->boot_service_name); i += 4) {
			memcpy(dword, &vsd->boot_service_name[i], 4);
			vsd->boot_service_name[i + 0] = dword[3];
			vsd->boot_service_name[i + 1] = dword[2];
			vsd->boot_service_name[i + 2] = dword[1];
			vsd->boot_service_name[i + 3] = dword[0];
		}
	}
}

/*
 * Mid level flash functions (firmware image)
 */

static int flash_image_read_from_file(char *fname,
	unsigned char **isbuf, unsigned int *isbufsz,
	unsigned char **psbuf, unsigned int *psbufsz,
	unsigned char **ibuf, unsigned int *ibufsz)
{
	unsigned int sector_sz = 0;
	uint32_t signature;
	unsigned char *buf;
	struct stat imgstat;
	FILE *fimg;

	/* Open and read image files */
	fimg = fopen(fname, "r");
	if (fimg == NULL) {
		fprintf(stderr, "tvflash: error opening file %s!\n", fname);
		return 1;
	}

	/* Get the size of the image */
	if (fstat(fileno(fimg), &imgstat) < 0) {
		fprintf(stderr, "tvflash: cannot do stat on firmware image!\n");
		return 1;
	}

	buf = malloc(imgstat.st_size);
	if (!buf) {
		fprintf(stderr, "tvflash: cannot allocate memory for the image buffer!\n");
		return 1;
	}

	if (fread(buf, imgstat.st_size, 1, fimg) != 1) {
		fprintf(stderr, "tvflash: error reading file %s!\n", fname);
		return 1;
	}
	fclose(fimg);

	/* Check the signature on the IS */
	signature = be32_to_cpu(*(uint32_t *)(buf + 0x24));
	if (signature == 0x5a445a44) {
		unsigned int is_sz;

		is_sz = be32_to_cpu(*(uint32_t *)(buf + 0x2c));

		if (is_sz > 4 && is_sz < 1048576) {
			uint16_t crc, img_crc;

			/* is_sz is a count of dwords */
			is_sz = is_sz * 4;
			img_crc = be32_to_cpu(*(uint32_t *)(buf + 0x28 + is_sz + 12));

			/* Verify the CRC of the IS */
			crc = flash_crc16(buf + 0x28, is_sz + 16 - 4);
			if (crc == img_crc) {
				unsigned int sector_sz_ptr, log2_sector_sz;

				/* Then grab the sector size */
				sector_sz_ptr	= be16_to_cpu(*(uint16_t *)(buf + 0x16));
				log2_sector_sz = be16_to_cpu(*(uint16_t *)(buf + sector_sz_ptr + 0x32));

				/*
				 * Do some sanity checking of the result.
				 * Anything less than 4KB or more than
				 * 1MB is suspicious and thrown out
				 */
				if (log2_sector_sz >= 12 && log2_sector_sz <= 20)
					sector_sz = 1 << log2_sector_sz;
			}
		}
	}

	/* Check for PPS */
	if (sector_sz &&
	    be32_to_cpu(*(uint32_t *)(buf + sector_sz + 8)) == 0x5a445a44) {
		union vsd *vsd;

		/* Failsafe firmware image */
		*isbuf = buf;
		*isbufsz = sector_sz;
		*psbuf = buf + sector_sz;
		*psbufsz = sector_sz;
		*ibuf = buf + be32_to_cpu(*(uint32_t *)(buf + sector_sz + 0));
		*ibufsz = be32_to_cpu(*(uint32_t *)(buf + sector_sz + 4));

		/* Perform some image fixups */
		vsd = (union vsd *)(buf + sector_sz + 0x20);
		if (be16_to_cpu(vsd->data.signature) == VSD_SIGNATURE_TOPSPIN &&
		    be16_to_cpu(vsd->data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN)
			fixup_topspin_vsd(&vsd->data.vendor.topspin);
	} else {
		/* Non failsafe firmware image */
		*isbuf = *psbuf = NULL;
		*isbufsz = *psbufsz = 0;
		*ibuf = buf;
		*ibufsz = imgstat.st_size;
	}

	return 0;
}

static int validate_xps(struct image *image, unsigned char *psbuf)
{
	/* Check for signature */
	image->valid = (be32_to_cpu(*(uint32_t *)(psbuf + 0x8)) == 0x5a445a44);
	if (image->valid) {
		image->addr = be32_to_cpu(*(uint32_t *)(psbuf + 0x0));
		image->size = be32_to_cpu(*(uint32_t *)(psbuf + 0x4));
		/* Check CRC for xPS */
		image->valid = (flash_crc16(psbuf, 0x104) == be16_to_cpu(*(uint16_t *)(psbuf + 0x106)));

		memcpy(image->vsd.raw, psbuf + 0x20, sizeof(image->vsd.raw));

		if (be16_to_cpu(image->vsd.data.signature) == VSD_SIGNATURE_TOPSPIN &&
		    be16_to_cpu(image->vsd.data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN)
			fixup_topspin_vsd(&image->vsd.data.vendor.topspin);
	}

	return image->valid;
}

static int flash_check_failsafe(struct tvdevice *tvdev)
{
	unsigned int sector_sz = 0;
	struct failsafe *failsafe = &tvdev->failsafe;
	uint32_t signature;
	unsigned char *psbuf;
	int i;

	/* Check the signature on the IS */
	signature = flash_read(tvdev, 0x24);
	if (signature == 0x5a445a44) {
		unsigned char *is;
		unsigned int is_sz;

		is_sz = flash_read(tvdev, 0x2c);
		if (is_sz > 4 && is_sz < 1048576) {
			uint16_t crc, img_crc;

			/* is_sz is a count of dwords */
			is_sz = is_sz * 4;
			is = malloc(is_sz + 16);
			if (!is) {
				fprintf(stderr, "couldn't allocate %d bytes for sector buffer\n",
					is_sz);
				exit(1);
			}

			for (i = 0; i < is_sz + 16; i += 4)
				*(uint32_t *)(is + i) = be32_to_cpu(flash_read(tvdev, 0x28 + i));

			img_crc = flash_read(tvdev, 0x28 + is_sz + 12);

			/* Verify the CRC of the IS */
			crc = flash_crc16(is, is_sz + 16 - 4);
			if (crc == img_crc) {
				unsigned int sector_sz_ptr, log2_sector_sz;

				/* Then grab the sector size */
				sector_sz_ptr	= flash_read(tvdev, 0x14) & 0xffff;
				log2_sector_sz = flash_read(tvdev, sector_sz_ptr + 0x30) & 0xffff;

				/*
				 * Do some sanity checking of the result.
				 * Anything less than 4KB or more than
				 * 1MB is suspicious and thrown out
				 */
				if (log2_sector_sz >= 12 && log2_sector_sz <= 20)
					sector_sz = 1 << log2_sector_sz;
			}

			free(is);
		}
	}

	if (!sector_sz)
		return 0;

	psbuf = malloc(sector_sz);
	if (!psbuf) {
		fprintf(stderr, "couldn't allocate temp buffer for PPS/SPS (size = %d)\n",
			sector_sz);
		exit(1);
	}

	/* Check both PPS and SPS for valid signatures */

	/* Read out the PPS */
	for (i = 0; i < sector_sz; i += 4)
		*(uint32_t *)(psbuf + i) = be32_to_cpu(flash_read(tvdev, sector_sz + i));

	validate_xps(&failsafe->images[0], psbuf);

	/* Read out the SPS */
	for (i = 0; i < sector_sz; i += 4)
		*(uint32_t *)(psbuf + i) = be32_to_cpu(flash_read(tvdev, sector_sz * 2 + i));

	validate_xps(&failsafe->images[1], psbuf);

	/*
	 * Last sanity check. We can't say that the IS is programmed correctly
	 * if we don't see a valid PPS or SPS.
	 */
	failsafe->valid = (failsafe->images[0].valid || failsafe->images[1].valid);

	free(psbuf);

	return failsafe->valid;
}

/* GUID section structure */
struct guid_section {
	uint8_t	gs_fw_reserved_1[16];
	uint8_t	gs_node_guid	[GUID_LEN];
	uint8_t	gs_port1_guid [GUID_LEN];
	uint8_t	gs_port2_guid [GUID_LEN];
	uint8_t	gs_sysimg_guid[8];
	uint8_t	gs_fw_reserved_2[2];
	uint16_t gs_crc16;
};

/* Get the GUIDs currently programmed in the flash */
static int flash_get_curr_guids(struct tvdevice *tvdev, unsigned char *guid_node,
	unsigned char *guid_port1, unsigned char *guid_port2)
{
	unsigned int imageaddr = 0;
	unsigned char *up;
	unsigned int paddr;
	int bcnt;

	if (tvdev->failsafe.valid) {
		if (tvdev->failsafe.images[0].valid)
			imageaddr = tvdev->failsafe.images[0].addr;
		else
			imageaddr = tvdev->failsafe.images[1].addr;
	} else
		imageaddr = 0;

	paddr = flash_read(tvdev, imageaddr + TV_FLASH_GUID_OFF) + imageaddr;
	if (paddr > tvdev->flash_size) {
		fprintf(stderr, "GUID offset (0x%X) is larger than flash size (0x%X)!\n",
			paddr, tvdev->flash_size);
		return 1;
	}

	/* Read the Node GUID */
	up = guid_node;
	for (bcnt = 0; bcnt < GUID_LEN; bcnt++)
		*up++ = flash_byte_read(tvdev, paddr++);

	/* Read the Port1 GUID */
	up = guid_port1;
	for (bcnt = 0; bcnt < GUID_LEN; bcnt++)
		*up++ = flash_byte_read(tvdev, paddr++);

	/* Read the Port2 GUID */
	up = guid_port2;
	for (bcnt = 0; bcnt < GUID_LEN; bcnt++)
		*up++ = flash_byte_read(tvdev, paddr++);

	return 0;
}

/* GUID update */
static int flash_guids_update(struct tvdevice *tvdev,
	unsigned char *ibuf, unsigned int ibufsz,
	unsigned char *guid_node, unsigned char *guid_port1,
	unsigned char *guid_port2)
{
	unsigned int paddr;
	struct guid_section *gsp;
	uint16_t new_crc;

	/* Endianess problems suck */
	paddr = be32_to_cpu(*(uint32_t *)(ibuf + TV_FLASH_GUID_OFF));
	if (paddr > ibufsz) {
		fprintf(stderr, "GUID pointer (0x%X) is larger than image size (0x%X)!\n",
			paddr, ibufsz);
		return 1;
	}

	/* Blend in the new GUIDs */
	memcpy(&ibuf[paddr], guid_node, GUID_LEN);
	memcpy(&ibuf[paddr + GUID_LEN], guid_port1, GUID_LEN);
	if (tvdev->board->num_ports > 1)
		memcpy(&ibuf[paddr + GUID_LEN * 2], guid_port2, GUID_LEN);

	printf("New Node  GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
		guid_node[0], guid_node[1], guid_node[2], guid_node[3],
		guid_node[4], guid_node[5], guid_node[6], guid_node[7]);
	printf("New Port1 GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
		guid_port1[0], guid_port1[1], guid_port1[2], guid_port1[3],
		guid_port1[4], guid_port1[5], guid_port1[6], guid_port1[7]);
	if (tvdev->board->num_ports > 1)
		printf("New Port2 GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
			guid_port2[0], guid_port2[1], guid_port2[2], guid_port2[3],
			guid_port2[4], guid_port2[5], guid_port2[6], guid_port2[7]);

	/* Get pointer to the GUID section in the image buffer */
	gsp = (struct guid_section *)(ibuf + paddr - sizeof(gsp->gs_fw_reserved_1));

	/* Recalculate GUID section CRC-16 - FW_Reserver_2 and CRC itself excluded */
	new_crc = flash_crc16((uint8_t *)gsp, (uint32_t)&((struct guid_section *)0)->gs_fw_reserved_2);

	gsp->gs_crc16 = cpu_to_be16(new_crc);

	return 0;
}

/*
 * High level flash functions (information, read/write firmware images)
 */

/* Identify */
static int create_ver_str(union vsd *vsd, char *buf, unsigned int buflen)
{
	if (vsd->data.vendor.topspin.revision_ver) {
		if (vsd->data.vendor.topspin.revision_ver & 0x80)
			snprintf(buf, buflen, "%d.%d.%03d-rc%d",
				vsd->data.vendor.topspin.major_ver,
				vsd->data.vendor.topspin.minor_ver,
				be16_to_cpu(vsd->data.vendor.topspin.micro_ver),
				vsd->data.vendor.topspin.revision_ver & ~(-1 << 7));
		else
			snprintf(buf, buflen, "%d.%d.%03d_%d",
				vsd->data.vendor.topspin.major_ver,
				vsd->data.vendor.topspin.minor_ver,
				be16_to_cpu(vsd->data.vendor.topspin.micro_ver),
				vsd->data.vendor.topspin.revision_ver);
	} else
		snprintf(buf, buflen, "%d.%d.%03d",
			vsd->data.vendor.topspin.major_ver,
			vsd->data.vendor.topspin.minor_ver,
			be16_to_cpu(vsd->data.vendor.topspin.micro_ver));

	return strlen(buf);
}

static void print_vpd_info(struct tvdevice *tvdev)
{
	char str[MAX_STR];
	unsigned short i = 0, rlen;
	unsigned char *ptr = tvdev->vpd.vpd_char;
	unsigned short len16;
	unsigned char len8 = 0, cksum = 0;
	int len = 0;
	int pn = 0, ec = 0, sn = 0, fp = 0, dc = 0, rv = 0;

	printf("\n  Vital Product Data\n");

	/* First field is string tag */
	if (ptr[0] != 0x82) {
		fprintf(stderr, "   Error. String Tag not present (found tag %02x instead)\n",
			ptr[0]);
		return;
	}

	/* print out String Tag */
	memcpy(&len16, ptr + 1, sizeof(len16));
	len16 = le16_to_cpu(len16);
	memcpy(str, &ptr[3], len16);
	str[len16] = 0;
	printf("    Product Name: %s\n", str);

	/* string len + 2 bytes of length field + 1 byte of tag */
	ptr = &ptr[len16 + 3];

	/* check for Read only tag */
	if (ptr[0] != 0x90) {
		fprintf(stderr, "    Error. R Tag not present\n");
		return;
	}
	memcpy(&rlen, ptr + 1, sizeof(rlen));	/* Read only resource len */
	rlen = le16_to_cpu(rlen);

	/* parse and print out each field, till we hit end tag */
	ptr += 3;		 /* 1 byte rtag + 2 bytes length field */
	while (i < rlen) {
		if ((ptr[0] == 'P') && (ptr[1] == 'N')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    P/N: %s\n", str);
			pn = 1;
		} else if ((ptr[0] == 'E') && (ptr[1] == 'C')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    E/C: %s\n", str);
			ec = 1;
		} else if ((ptr[0] == 'S') && (ptr[1] == 'N')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    S/N: %s\n", str);
			sn = 1;
		} else if ((ptr[0] == 'V') && (ptr[1] == '0')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    Freq/Power: %s\n", str);
			fp = 1;
		} else if ((ptr[0] == 'V') && (ptr[1] == '2')) {
			len8 = ptr[2];
			memcpy(str, &ptr[3], len8);
			str[len8] = 0;
			printf("    Date Code: %s\n", str);
			dc = 1;
		} else if ((ptr[0] == 'R') && (ptr[1] == 'V')) {
			len = &ptr[2] - tvdev->vpd.vpd_char + 1;

			while (len >= 0) {
				cksum += tvdev->vpd.vpd_char[len];
				len--;
			}

			if (cksum == 0)
				printf("    Checksum: Ok\n");
			else
				printf("    Checksum: Incorrect\n");

			rv = 1;
		} else if (ptr[0] == 0x78)	/* End Tag */
			 break;

		i += (len8 + 3);
		ptr += (len8 + 3);
	}
	if (!pn)
		printf("    P/N: N/A\n");
	if (!ec)
		printf("    E/C: N/A\n");
	if (!sn)
		printf("    S/N: N/A\n");
	if (!dc)
		printf("    Date Code: N/A\n");
	if (!fp)
		printf("    Freq/Power: N/A\n");
	if (!rv)
		printf("    Checksum: N/A\n");
}

static int open_device(struct tvdevice *tvdev)
{
#if defined(__PPC__)
	unsigned short command;

	/* Enable memory regions if it's disabled */
	command = pci_read_word(tvdev->pdev, 0x04);
	if (!(command & 0x02))
		pci_write_word(tvdev->pdev, 0x04, command | 0x02);
#endif

	if (!use_config_space && !(tvdev->flags & FLAG_RECOVERY)) {
		int fd;

		fd = open("/dev/mem", O_RDWR, O_SYNC);
		if (fd >= 0) {
			tvdev->bar0 = mmap64(NULL, 1 << 20, PROT_READ | PROT_WRITE, MAP_SHARED,
					 fd, tvdev->pdev->base_addr[0] & PCI_ADDR_MEM_MASK);
			close(fd);
			if (tvdev->bar0 == MAP_FAILED)
				tvdev->bar0 = NULL;
		} else
			tvdev->bar0 = NULL;
	} else
		tvdev->bar0 = NULL;

	if (tvdev->bar0) {
		tvdev->method = METHOD_MMAP;
		return 0;
	}

#if 0
	/* FIXME: Test that we can do this first */
	tvdev->method = METHOD_PCI_CFG;

	return 0;
#else
	return 1;
#endif
}

static void identify_flash_device(struct tvdevice *tvdev)
{
	char cfiq[1024], str[4];
	int i, width = 0;

	switch (tvdev->board->chip) {
	case CHIP_SINAI:
		/* SPI device */
		spi_discover_serial_proms(tvdev);
		break;

	default:
		tvdev->flash_size = 0x400000;	/* 4MB */
		tvdev->flash_bank_shift = 19;
		tvdev->flash_bank_mask = (1 << tvdev->flash_bank_shift) - 1;

		/* Parallel flash chip */

		/* Get CFI info for flash device */
		flash_write_cmd(tvdev, 0x55, 0xFF);	/* Reset */
		flash_write_cmd(tvdev, 0x55, 0x98);	/* CFI Query */

		str[0] = flash_byte_read(tvdev, 0x10);
		str[1] = flash_byte_read(tvdev, 0x11);
		str[2] = flash_byte_read(tvdev, 0x12);
		str[3] = 0;

		if (strcmp(str, "QRY") == 0)
			width = 1;
		else {
			str[0] = flash_byte_read(tvdev, 0x20);
			str[1] = flash_byte_read(tvdev, 0x22);
			str[2] = flash_byte_read(tvdev, 0x24);
			str[3] = 0;

			if (strcmp(str, "QRY") == 0)
				width = 2;
		}

		if (!width) {
			fprintf(stderr, "CFI query failed. Unknown flash device.\n");
			exit(1);
		}

		for (i = 0; i < sizeof(cfiq); i++)
			cfiq[i] = flash_byte_read(tvdev, i * width);

		tvdev->flash_command_set = cfiq[0x13];

		/*
		 * FIXME: This is a hack for now. We should really get this
		 * from the CFI query. Careful, MX chips have buggy
		 * information.
		 */
		switch (tvdev->flash_command_set) {
		case CS_INTEL:
			tvdev->flash_sector_sz = 0x20000;	/* 128KB */
			break;
		case CS_AMD:
			tvdev->flash_sector_sz = 0x10000;	/* 64KB */
			break;
		default:
			printf("Unknown flash command set.\n");
			exit(1);
		}

		break;
	}

	tvdev->flash_bank_mask = (1 << tvdev->flash_bank_shift) - 1;

	flash_chip_reset(tvdev);

	tvdev->flash_bank = -1;
	flash_set_bank(tvdev, 0);
}

static int open_hca(struct tvdevice *tvdev)
{
	int ret;

	ret = open_device(tvdev);
	if (ret)
		return ret;

	read_cfg(tvdev, 0xF0150);
	write_cfg(tvdev, 0xF0150, 1 << 30);

	grab_gpio(tvdev);

	tvdev->board = tvdev->pciid->identify(tvdev);
	if (!tvdev->board) {
		fprintf(stderr, "unable to identify board\n");
		return 1;
	}

	switch (tvdev->board->chip) {
	case CHIP_SINAI:
		/* SPI flash */

		/* Unlock GPIO */
		write_cfg(tvdev, GPIO_LOCK, 0xAAAA);

#define SPI_ENABLE	((1 << (NUM_SPIS - 1)) - 1)
		write_cfg(tvdev, GPIO_DIR + 4, SPI_ENABLE << 5);
		write_cfg(tvdev, GPIO_POL + 4, (SPI_ENABLE ^ 0x07) << 5);
		write_cfg(tvdev, GPIO_MOD + 4, (SPI_ENABLE ^ 0x07) << 5);
		break;
	default:
		/* Parallel flash */

		/* Set the direction of the flash pins to output */
		write_cfg(tvdev, GPIO_DIR + 4, read_cfg(tvdev, GPIO_DIR + 4) | (0x07 << 4));

		/* Clear the data for the flash pins to start at bank 0 */
		write_cfg(tvdev, 0xF0080 + 0x54, 0x07 << 4);

		/* Clear the polarity for the flash pins */
		write_cfg(tvdev, GPIO_POL + 4, read_cfg(tvdev, GPIO_POL + 4) & ~(0x07 << 4));

		/* Clear the output mode for the flash pins */
		write_cfg(tvdev, GPIO_MOD + 4, read_cfg(tvdev, GPIO_MOD + 4) & ~(0x07 << 4));
		break;
	}

	identify_flash_device(tvdev);

	if (tvdev->flash_command_set == CS_UNKNOWN) {
		printf("Unknown flash device, cannot continue\n");
		return 1;
	}

	/* RevC LionCub boards use Intel flash chips and use a different firmware */
	if (tvdev->board == &lioncub && tvdev->flash_command_set == CS_INTEL)
		tvdev->board = &lioncub_revc;

	return 0;
}

static void close_hca(struct tvdevice *tvdev)
{
	release_gpio(tvdev);

	if (tvdev->bar0)
		munmap(tvdev->bar0, 1 << 20);
}

static int identify_hca(int num, struct tvdevice *tvdev,
		 enum identify_mode identify_mode, int *unreliable)
{
	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", num);
		return 1;
	}

	flash_check_failsafe(tvdev);

	switch (identify_mode) {
	case IDENTIFY_EXTENDED:
		printf("HCA #%d: ", num);
		switch (tvdev->board->chip) {
		case CHIP_TAVOR:
			printf("MT23108");
			break;
		case CHIP_ARBEL:
			printf("MT25208");
			if (tvdev->flags & FLAG_TAVOR_COMPAT)
				printf(" Tavor Compat");
			break;
		case CHIP_SINAI:
			printf("MT25204");
			break;
		default:
			printf("Unknown");
			break;
		}

		if (tvdev->flags & FLAG_RECOVERY)
			printf(" (recovery mode)");

		printf(", %s", tvdev->board->name);

		if (tvdev->flags & FLAG_RECOVERY) {
			printf("(*)");
			*unreliable = 1;
		}

		printf(", revision %02X\n", tvdev->revision);

		if (tvdev->failsafe.valid) {
			union vsd *vsd;

			if (tvdev->failsafe.images[0].valid) {
				vsd = &tvdev->failsafe.images[0].vsd;
				if (be16_to_cpu(vsd->data.signature) == VSD_SIGNATURE_TOPSPIN &&
				    be16_to_cpu(vsd->data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
					char ver_str[40];

					create_ver_str(vsd, ver_str, sizeof(ver_str));
					if (vsd->data.vendor.topspin.build_rev[0])
						printf("  Primary image is v%s build %s.%d, with label '%s'\n",
							ver_str,
							vsd->data.vendor.topspin.build_rev,
							le16_to_cpu(tvdev->failsafe.images[0].vsd.data.vendor.topspin.build_num),
							vsd->data.vendor.topspin.hw_label);
					else
						printf("  Primary image is v%s, with label '%s'\n",
							ver_str,
							vsd->data.vendor.topspin.hw_label);
				} else
					printf("  Primary image is valid, unknown source\n");

				if (verbose > 1)
					printf("    0x%08x bytes @ 0x%08x\n",
						tvdev->failsafe.images[0].size,
						tvdev->failsafe.images[0].addr);
			} else
				printf("  Primary image is NOT valid\n");

			if (tvdev->failsafe.images[1].valid) {
				vsd = &tvdev->failsafe.images[1].vsd;
				if (be16_to_cpu(vsd->data.signature) == VSD_SIGNATURE_TOPSPIN &&
				    be16_to_cpu(vsd->data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
					char ver_str[40];

					create_ver_str(vsd, ver_str, sizeof(ver_str));
					if (vsd->data.vendor.topspin.build_rev[0])
						printf("  Secondary image is v%s build %s.%d, with label '%s'\n",
							ver_str,
							vsd->data.vendor.topspin.build_rev,
							le16_to_cpu(vsd->data.vendor.topspin.build_num),
							vsd->data.vendor.topspin.hw_label);
					else
						printf("  Secondary image is v%s, with label '%s'\n",
							ver_str,
							vsd->data.vendor.topspin.hw_label);
				} else
					printf("  Secondary image is valid, unknown source\n");

				if (verbose > 1)
					printf("    0x%08x bytes @ 0x%08x\n",
						tvdev->failsafe.images[1].size,
						tvdev->failsafe.images[1].addr);
			} else
				printf("  Secondary image is NOT valid\n");
		} else
			printf("  Firmware is NOT installed in failsafe mode\n");

		/* If vpd is present, read it and print it */
		if (tvdev->vpd_present)
			print_vpd_info(tvdev);
		break;
	case IDENTIFY_PRIMARY_FIRMWARE_LABEL:
		if (tvdev->failsafe.valid) {
			union vsd *vsd;

			if (tvdev->failsafe.images[0].valid) {
				vsd = &tvdev->failsafe.images[0].vsd;
				if (be16_to_cpu(vsd->data.signature) == VSD_SIGNATURE_TOPSPIN &&
				    be16_to_cpu(vsd->data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
					unsigned int build_major = 0, build_minor = 0, build_micro = 0;

					sscanf(vsd->data.vendor.topspin.build_rev, "%u.%u.%u",
						 &build_major, &build_minor, &build_micro);

					printf("%s:%d:%d:%d:%d:%d:%d:%d:%d:%d\n",
						vsd->data.vendor.topspin.hw_label,
						(vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_AUTOUPGRADE)) ? 1 : 0,
						vsd->data.vendor.topspin.major_ver,
						vsd->data.vendor.topspin.minor_ver,
						be16_to_cpu(vsd->data.vendor.topspin.micro_ver),
						vsd->data.vendor.topspin.revision_ver,
						build_major, build_minor, build_micro,
						le16_to_cpu(vsd->data.vendor.topspin.build_num));
				} else
					printf("Unknown\n");
			} else
				printf("Invalid\n");
		} else
			printf("Invalid\n");

		break;
	case IDENTIFY_HARDWARE_LABEL:
		/* Print out what we think the firmware label should be */
		if (tvdev->board->fwlabel)
			printf("%s.%02X",
				tvdev->board->fwlabel, tvdev->revision);
		else
			printf("Unknown.%02X", tvdev->revision);
		break;
	}

	close_hca(tvdev);

	return 0;
}

static int identify_hcas(int hca, enum identify_mode identify_mode)
{
	struct tvdevice *tvdev;
	int ret = 0, count = 0, unreliable = 0;

	if (hca >= 0) {
		tvdev = find_device(hca);
		if (!tvdev) {
			fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n",
				hca);
			return 1;
		}

		ret = identify_hca(hca, tvdev, identify_mode, &unreliable);
	} else {
		for (tvdev = tvdevices; tvdev; tvdev = tvdev->next) {
			ret = identify_hca(count++, tvdev, identify_mode, &unreliable);
			if (ret)
				break;
		}
	}

	if (unreliable)
		printf("\n(*) Unreliable in recovery mode\n");

	return ret;
}

static int identify_firmware(char *ifname, enum identify_mode identify_mode)
{
	unsigned char *isbuf, *psbuf = NULL, *ibuf;
	unsigned int isbufsz, psbufsz, ibufsz;
	union vsd *vsd = NULL;

	/* Read the flash image into the memory */
	if (flash_image_read_from_file(ifname, &isbuf, &isbufsz, &psbuf,
					&psbufsz, &ibuf, &ibufsz))
		return 1;

	vsd = (union vsd *)(psbuf + 0x20);

	switch (identify_mode) {
	case IDENTIFY_EXTENDED:
		printf("Firmware image %s", ifname);

		if (psbuf && vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_AUTOUPGRADE))
			printf(" (firmware autoupgrade)");

		printf("\n");

		if (psbuf) {
			if (be16_to_cpu(vsd->data.signature) == VSD_SIGNATURE_TOPSPIN &&
			    be16_to_cpu(vsd->data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
				char ver_str[40];

				create_ver_str(vsd, ver_str, sizeof(ver_str));
				if (vsd->data.vendor.topspin.build_rev[0])
					printf("  Image is v%s build %s.%d, with label '%s'\n",
						ver_str, vsd->data.vendor.topspin.build_rev,
						le16_to_cpu(vsd->data.vendor.topspin.build_num),
						vsd->data.vendor.topspin.hw_label);
				else
					printf("  Image is v%s, with label '%s'\n",
						ver_str,
						vsd->data.vendor.topspin.hw_label);
			} else
				printf("  Image is valid, unknown source\n");
		} else
			printf("  Image is NOT valid\n");

		break;
	case IDENTIFY_PRIMARY_FIRMWARE_LABEL:
		if (psbuf) {
			if (be16_to_cpu(vsd->data.signature) == VSD_SIGNATURE_TOPSPIN &&
			    be16_to_cpu(vsd->data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
				unsigned int build_major = 0, build_minor = 0, build_micro = 0;

				sscanf(vsd->data.vendor.topspin.build_rev, "%u.%u.%u",
					 &build_major, &build_minor, &build_micro);

				printf("%s:%d:%d:%d:%d:%d:%d:%d:%d:%d\n",
					 vsd->data.vendor.topspin.hw_label,
					 (vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_AUTOUPGRADE)) ? 1 : 0,
					 vsd->data.vendor.topspin.major_ver,
					 vsd->data.vendor.topspin.minor_ver,
					 be16_to_cpu(vsd->data.vendor.topspin.micro_ver),
					 vsd->data.vendor.topspin.revision_ver,
					 build_major, build_minor, build_micro,
					 le16_to_cpu(vsd->data.vendor.topspin.build_num));
			} else
				printf("Unknown\n");
		} else
			printf("Invalid\n");

		break;

	default:
		printf("Invalid mode\n");
		break;
	}

	return 0;
}

static int print_hca_guids(int num, struct tvdevice *tvdev)
{
	unsigned char guid_node[GUID_LEN];
	unsigned char guid_port1[GUID_LEN];
	unsigned char guid_port2[GUID_LEN];

	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", num);
		return 1;
	}

	flash_check_failsafe(tvdev);

	if (flash_get_curr_guids(tvdev, guid_node, guid_port1, guid_port2)) {
		fprintf(stderr, "Cannot determine previous GUID. Corrupted flash?\n");
		return 1;
	}

	printf("HCA #%d\n", num);
	printf("Node  GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
		guid_node[0], guid_node[1], guid_node[2], guid_node[3],
		guid_node[4], guid_node[5], guid_node[6], guid_node[7]);
	printf("Port1 GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
		guid_port1[0], guid_port1[1], guid_port1[2], guid_port1[3],
		guid_port1[4], guid_port1[5], guid_port1[6], guid_port1[7]);
	if (tvdev->board->num_ports > 1)
		printf("Port2 GUID = %02x%02x%02x%02x%02x%02x%02x%02x\n",
			guid_port2[0], guid_port2[1], guid_port2[2], guid_port2[3],
			guid_port2[4], guid_port2[5], guid_port2[6], guid_port2[7]);

	close_hca(tvdev);

	return 0;
}

static int print_guids(int hca)
{
	struct tvdevice *tvdev;
	int ret = 0, count = 0;

	if (hca >= 0) {
		tvdev = find_device(hca);
		if (!tvdev) {
			fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n", hca);
			return 1;
		}

		ret = print_hca_guids(hca, tvdev);
	} else {
		for (tvdev = tvdevices; tvdev; tvdev = tvdev->next) {
			ret = print_hca_guids(count++, tvdev);
			if (ret)
				break;
		}
	}

	return ret;
}

/* Download (from HCA to host) */
static int flash_image_write_to_file(struct tvdevice *tvdev, char *fname)
{
	char *buffer;
	int i, fd;
	unsigned int offset;

	buffer = malloc(tvdev->flash_sector_sz);
	if (!buffer) {
		fprintf(stderr, "couldn't allocated %d bytes of memory for buffer\n",
			tvdev->flash_sector_sz);
		return 1;
	}

	if (strcmp(fname, "-") == 0)
		fd = fileno(stdout);
	else
		fd = open(fname, O_RDWR | O_CREAT | O_TRUNC, 0644);

	if (fd < 0) {
		fprintf(stderr, "couldn't open %s to save firmware: %m\n", fname);
		return 1;
	}

	offset = 0;
	while (offset < tvdev->flash_size) {
		for (i = 0; i < tvdev->flash_sector_sz; i += 4) {
			unsigned int data;

			data = flash_read(tvdev, offset + i);

			buffer[i + 0] = (data >> 24) & 0xff;
			buffer[i + 1] = (data >> 16) & 0xff;
			buffer[i + 2] = (data >> 8) & 0xff;
			buffer[i + 3] = data & 0xff;
		}

		write(fd, buffer, tvdev->flash_sector_sz);

		offset += tvdev->flash_sector_sz;
	}

	close(fd);

	return 0;
}

static int download_firmware(int hca, char *ofname)
{
	struct tvdevice *tvdev;
	int ret;

	tvdev = find_device(hca);
	if (!tvdev) {
		fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n", hca);
		return 1;
	}

	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", hca);
		return 1;
	}

	flash_check_failsafe(tvdev);

	ret = flash_image_write_to_file(tvdev, ofname);

	close_hca(tvdev);

	return ret;
}

static void flash_write_block(struct tvdevice *tvdev, unsigned char status,
	unsigned int addr, unsigned char *buffer, unsigned int buflen)
{
	unsigned int pos = 0, bufsz = buflen;

	while (bufsz > 0) {
		unsigned int len = 16;
		int i;

		flash_set_bank(tvdev, addr);

		if (len > bufsz)
			bufsz = len;

		switch (tvdev->flash_command_set) {
		case CS_SPI:
			flash_spi_write_block(tvdev, addr, buffer + pos, len);
			break;
		default:
			for (i = 0; i < len; i++)
				flash_write_byte(tvdev, addr + i, buffer[pos + i]);
			break;
		}
		addr += len;
		pos += len;
		bufsz -= len;

		if (status)
			status_update(status, pos);
	}
}

/* Upload (from host to HCA) */
static int flash_compare_invariant_sector(struct tvdevice *tvdev, unsigned char *isbuf,
	unsigned int isbufsz)
{
	unsigned int addr, data;

	flash_chip_reset(tvdev);

	for (addr = 0; addr < isbufsz; addr += 4) {
		int i;

		data = flash_read(tvdev, addr);
		for (i = 0; i < 4; i++) {
			if (isbuf[addr + i] != ((data >> ((3 - i) * 8)) & 0xFF))
				return 1;
		}
	}

	return 0;
}

static void flash_move_pps_to_sps(struct tvdevice *tvdev)
{
	unsigned int addr;
	unsigned char *sector;

	flash_chip_reset(tvdev);

	sector = malloc(tvdev->flash_sector_sz);
	if (!sector) {
		fprintf(stderr, "Unable to allocate %d bytes of memory for buffer\n",
			tvdev->flash_sector_sz);
		exit(1);
	}

	for (addr = 0; addr < tvdev->flash_sector_sz; addr += 4) {
		uint32_t data;
		int i;

		data = flash_read(tvdev, tvdev->flash_sector_sz + addr);
		for (i = 0; i < 4; i++)
			sector[addr + i] = ((data >> ((3 - i) * 8)) & 0xFF);
	}

	/* Erase the SPS sector */
	status_update('E', 0);
	flash_erase_sector(tvdev, tvdev->flash_sector_sz * 2);

	/* Write the old PPS as the SPS */
	flash_write_block(tvdev, 'F', tvdev->flash_sector_sz * 2, sector, tvdev->flash_sector_sz);
	status_mark();

	flash_chip_reset(tvdev);

	free(sector);
}

static void flash_write_invariant_sector(struct tvdevice *tvdev, unsigned char *isbuf,
	unsigned int isbufsz)
{
	flash_chip_reset(tvdev);

	status_update('E', 0);
	flash_erase_sector(tvdev, 0);
	flash_write_block(tvdev, 'I', 0, isbuf, isbufsz);
	status_mark();

	flash_chip_reset(tvdev);
}

static void flash_write_pps(struct tvdevice *tvdev, unsigned int psbufoff,
	unsigned char *psbuf, unsigned int psbufsz, unsigned int ibufoff,
	unsigned int ibufsz)
{
	/* Set the image offset and size */
	*(uint32_t *)(psbuf + 0x0) = cpu_to_be32(ibufoff);
	*(uint32_t *)(psbuf + 0x4) = cpu_to_be32(ibufsz);

	/* Set signature and CRC to all-ones for now */
	*(uint32_t *)(psbuf + 0x8) = 0xFFFFFFFF;
	*(uint16_t *)(psbuf + 0x106) = 0xFFFF;

	/* Erase then write the PPS */
	flash_chip_reset(tvdev);

	status_update('E', 0);
	flash_erase_sector(tvdev, psbufoff);
	flash_write_block(tvdev, 'P', psbufoff, psbuf, psbufsz);
	status_mark();

	flash_chip_reset(tvdev);
}

static void flash_finish_failsafe(struct tvdevice *tvdev, unsigned int psbufoff,
	unsigned char *psbuf, unsigned int psbufsz)
{
	/* Set signature and CRC to something valid */
	*(uint32_t *)(psbuf + 0x8) = cpu_to_be32(0x5a445a44);
	*(uint16_t *)(psbuf + 0x106) = cpu_to_be16(flash_crc16(psbuf, 0x104));

	/* Write CRC and signature */
	flash_chip_reset(tvdev);

	flash_write_block(tvdev, 0, psbufoff + 0x106, psbuf + 0x106, sizeof(uint16_t));
	flash_write_block(tvdev, 0, psbufoff + 0x8, psbuf + 0x8, sizeof(uint32_t));

	flash_chip_reset(tvdev);
}

static void flash_write_image(struct tvdevice *tvdev, unsigned int ibufoff,
	unsigned char *ibuf, unsigned int ibufsz)
{
	unsigned int addr;

	/* Erase then write the firmware image */
	for (addr = 0; addr < ibufsz; addr += tvdev->flash_sector_sz) {
		unsigned int len = tvdev->flash_sector_sz;

		if (len > ibufsz - addr)
			len = ibufsz - addr;

		status_update('E', 0);
		flash_erase_sector(tvdev, ibufoff + addr);
		flash_write_block(tvdev, 'W', ibufoff + addr, ibuf + addr, len);
		status_mark();
	}

	flash_chip_reset(tvdev);
}

static unsigned int flash_verify_image(struct tvdevice *tvdev, unsigned int off,
	unsigned char *buf, unsigned int bufsz)
{
	unsigned int addr, data;

	flash_chip_reset(tvdev);

	for (addr = 0; addr < bufsz; addr += 4) {
		int i;

		status_update('V', addr);

		data = flash_read(tvdev, off + addr);
		for (i = 0; i < 4; i++) {
			if (buf[addr + i] != ((data >> ((3 - i) * 8)) & 0xFF))
				return addr + i;
		}
	}
	status_mark();

	return addr;
}

static int upload_firmware(int hca, char *ifname, char *guid, char *change_option_cmd)
{
	struct tvdevice *tvdev;
	unsigned char *isbuf, *psbuf, *ibuf;
	unsigned int isbufsz, psbufsz, ibufsz, addr, status_count;
	unsigned char new_guid_node[GUID_LEN], new_guid_port1[GUID_LEN],
	new_guid_port2[GUID_LEN];
	int ret = 1, sector = 0, program_is = 0;

	tvdev = find_device(hca);
	if (!tvdev) {
		fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n", hca);
		return 1;
	}

	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", hca);
		return 1;
	}

	flash_check_failsafe(tvdev);

	if (guid) {
		/* GUIDs given to us on command line */
		if (parse_guid(guid, new_guid_node)) {
			fprintf(stderr, "Unable to parse GUID from command line\n\n");
			usage();
		}

		inc_guid(new_guid_node, new_guid_port1);
		if (tvdev->board->num_ports > 1)
			inc_guid(new_guid_port1, new_guid_port2);
	} else {
		/* Preserve the current GUIDs */
		if (flash_get_curr_guids(tvdev, new_guid_node, new_guid_port1,
					 new_guid_port2)) {
			fprintf(stderr, "Cannot determine previous GUID. Corrupted flash?\n");
			fprintf(stderr, "GUID must be specified on command line.\n\n");
			usage();
		}

		/*
		 * Now we check the GUIDs to make sure they are valid or
		 * could possibly cause some confusion (the default Mellanox
		 * GUID)
		 */
		if (memcmp(new_guid_node,  "\x00\x00\x00\x00\x00\x00\x00\x00", 8) == 0 ||
		    memcmp(new_guid_port1, "\x00\x00\x00\x00\x00\x00\x00\x00", 8) == 0 ||
		    (tvdev->board->num_ports > 1 &&
		     memcmp(new_guid_port2, "\x00\x00\x00\x00\x00\x00\x00\x00", 8) == 0)) {
			fprintf(stderr, "GUIDs read from flash are all zeros. Corrupted flash?\n");
			fprintf(stderr, "GUID must be specified on command line.\n\n");
			usage();
		}

		if (memcmp(new_guid_node,  "\xff\xff\xff\xff\xff\xff\xff\xff", 8) == 0 ||
		    memcmp(new_guid_port1, "\xff\xff\xff\xff\xff\xff\xff\xff", 8) == 0 ||
		    (tvdev->board->num_ports > 1 &&
		     memcmp(new_guid_port2, "\xff\xff\xff\xff\xff\xff\xff\xff", 8) == 0)) {
			fprintf(stderr, "GUIDs read from flash are all ones. Device access error?\n");
			fprintf(stderr, "GUID must be specified on command line.\n\n");
			usage();
		}

		if ((memcmp(new_guid_node,  "\x00\x02\xc9\x00\x01\x00\xd0\x50", 8) == 0 ||
		     memcmp(new_guid_port1, "\x00\x02\xc9\x00\x01\x00\xd0\x51", 8) == 0 ||
		     (tvdev->board->num_ports > 1 &&
		      memcmp(new_guid_port2, "\x00\x02\xc9\x00\x01\x00\xd0\x52", 8) == 0)) &&
		    !force) {
			fprintf(stderr, "GUIDs read from flash are vendor default GUIDs. It is not recommended\n");
			fprintf(stderr, "that these GUIDs be used because of a high chance of GUID conflict.\n");
			fprintf(stderr, "Please specify new GUID on command line, or use -f option.\n");
			usage();
		}
	}

	/* Read the flash image into the memory */
	if (flash_image_read_from_file(ifname, &isbuf, &isbufsz, &psbuf,
				       &psbufsz, &ibuf, &ibufsz))
		return 1;

	if (psbuf) {
		union vsd *vsd;

		vsd = (union vsd *)(psbuf + 0x20);

		if (be16_to_cpu(vsd->data.signature) == VSD_SIGNATURE_TOPSPIN &&
		    be16_to_cpu(vsd->data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN) {
			union vsd *fvsd = &tvdev->failsafe.images[0].vsd;

			/* Update the time the image was flashed */
			vsd->data.vendor.topspin.flashtime = cpu_to_be32(time(NULL));

			/* Set default value, we'll overwrite this if we need to */
			vsd->data.vendor.topspin.flags |= cpu_to_be32(VSD_FLAG_AUTOUPGRADE);

			/* Copy over boot ROM options from old VSD */
			vsd->data.vendor.topspin.flags &= ~cpu_to_be32(VSD_FLAG_BOOT_OPTIONS);
			vsd->data.vendor.topspin.flags |= (fvsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_OPTIONS));

			if (vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_OPTIONS)) {
				vsd->data.vendor.topspin.flags &= ~cpu_to_be32(VSD_FLAG_BOOT_OPTIONS_MASK);
				vsd->data.vendor.topspin.flags |= (fvsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_OPTIONS_MASK));

				vsd->data.vendor.topspin.boot_version = fvsd->data.vendor.topspin.boot_version;
				vsd->data.vendor.topspin.boot_port = fvsd->data.vendor.topspin.boot_port;
				vsd->data.vendor.topspin.boot_ioc_num = fvsd->data.vendor.topspin.boot_ioc_num;
				memcpy(vsd->data.vendor.topspin.boot_dgid,
					fvsd->data.vendor.topspin.boot_dgid,
					sizeof(vsd->data.vendor.topspin.boot_dgid));
				memcpy(vsd->data.vendor.topspin.boot_service_name,
					fvsd->data.vendor.topspin.boot_service_name,
					sizeof(vsd->data.vendor.topspin.boot_service_name));
				vsd->data.vendor.topspin.boot_pxe_secs = fvsd->data.vendor.topspin.boot_pxe_secs;
			} else {
				/* Set some sensible defaults */
				vsd->data.vendor.topspin.flags |= cpu_to_be32(VSD_FLAG_BOOT_OPTIONS |
					 VSD_FLAG_BOOT_ENABLE_PORT_1 |
					 VSD_FLAG_BOOT_ENABLE_PORT_2 |
					 VSD_FLAG_BOOT_ENABLE_SCAN |
					 VSD_FLAG_BOOT_TYPE_WELL_KNOWN);
				vsd->data.vendor.topspin.boot_version = BOOT_VERSION;
				vsd->data.vendor.topspin.boot_pxe_secs = BOOT_PXE_SECS_DEFAULT;
			}

			if (change_option_cmd &&
			    parse_change_options(&vsd->data.vendor.topspin, change_option_cmd))
				return 1;

			/* Calculate the VSD checksum */
			vsd->data.checksum = 0;
			vsd->data.checksum = cpu_to_be16(flash_crc16(vsd->raw, sizeof(vsd->raw)));
		}

		/* PPS checksum will be updated as a final step */

		if (be16_to_cpu(vsd->data.signature) == VSD_SIGNATURE_TOPSPIN &&
		    be16_to_cpu(vsd->data.vendor.topspin.signature2) == VSD_SIGNATURE_TOPSPIN &&
		    !force) {
			if (tvdev->board->fwlabel) {
				char hwlabel[64];

				/*
				 * Check to make sure this file is correct for
				 * the hardware. We check only as far as the
				 * hwlabel since boot firmware will have extra
				 * information appended to the label.
				 */
				snprintf(hwlabel, sizeof(hwlabel), "%s.%02X",
					tvdev->board->fwlabel, tvdev->revision);
				if (strncasecmp(vsd->data.vendor.topspin.hw_label, hwlabel, strlen(hwlabel)) != 0) {
					fprintf(stderr, "Firmware image specified has hardware label '%s', but hardware\n",
						vsd->data.vendor.topspin.hw_label);
					fprintf(stderr, "is of type '%s'. Please confirm you are using the correct\n",
						hwlabel);
					fprintf(stderr, "firmware or use the force option (-f).\n");
					return 1;
				}
			} else {
				if (tvdev->flags & FLAG_RECOVERY) {
					fprintf(stderr, "WARNING: Unable to verify firmware image is appropriate for hardware when\n");
					fprintf(stderr, "hardware is in flash recovery mode.\n");
				} else
					fprintf(stderr, "WARNING: Unable to verify firmware image is appropriate for unknown hardware.\n");

				fprintf(stderr, "Will upload flash image in 20 seconds or hit Ctrl-C to exit.\n");
				sleep(20);
			}
		}

		if (tvdev->failsafe.valid) {
			if (tvdev->flash_command_set != CS_SPI ||
			    tvdev->flash_spi_sp_type == SP_ST_M25P80) {
			/*
			 * The mainstream case - We'll always use the
			 * space opposite of the PPS.
			 */ 
				if (tvdev->failsafe.images[0].addr + tvdev->failsafe.images[0].size < tvdev->flash_size / 2)
					sector = (tvdev->flash_size / 2) / tvdev->flash_sector_sz;
				else if (tvdev->failsafe.images[0].addr >= tvdev->flash_sector_sz * 3 + ibufsz)
					sector = 3;
				else {
					fprintf(stderr, "Unable to fit new image (size 0x%x) on flash in Failsafe mode\n", ibufsz);
					exit(1);
				}
			} else {
			/*
			 * Sinai with 2MB single flash prom has issue
			 * if the PPS points above 1MB
			 */
				if ((tvdev->failsafe.images[0].addr / tvdev->flash_sector_sz) == 3)
					sector = ((tvdev->flash_sector_sz * 3 + tvdev->failsafe.images[1].size) / tvdev->flash_sector_sz) + 1;
				else if (tvdev->failsafe.images[0].addr > tvdev->flash_sector_sz * 3)
					sector = 3;
				else {
					fprintf(stderr, "Unable to fit new image (size 0x%x) on flash in Failsafe mode\n", ibufsz);
					exit(1);
				}
			}
		} else
			/* No Invariant Sector yet (will be programmed below) */
			sector = 3;
	} else if (ibufsz > tvdev->flash_size) {
		fprintf(stderr, "Image size is larger than size of flash (0x%X)\n",
			tvdev->flash_size);
		return 1;
	}

	if (tvdev->failsafe.valid && !psbuf) {
		if (!force) {
			/* Flash is in Failsafe mode, but image doesn't have Failsafe information */
			printf("Flash is in failsafe mode, but image is not a failsafe firmware image.\n");
			printf("Please use a failsafe firmware image, or use the force option if you\n");
			printf("are sure you know what you are doing.\n");
			exit(1);
		} else {
			printf("WARNING: Downgrading flash from Failsafe to non Failsafe. Please hit Ctrl-C\n");
			printf("now if that is not what you wanted.\n");
			sleep(5);
		}
	}

	if (flash_guids_update(tvdev, ibuf, ibufsz, new_guid_node,
	new_guid_port1, new_guid_port2))
		return 1;

	/* Compare the currently programmed IS with the IS from the file */
	if (isbuf) {
		if (force > 1) {
			/* User specified -ff, always upgrade IS */
			if (tvdev->failsafe.valid) {
				/* But warn if the firmware is already in failsafe mode, just in case */
				printf("WARNING: Flash reprogramming won't be failsafe, continuing in 10 seconds\n");
				sleep(10);
			}

			program_is = 1;
		} else if (flash_compare_invariant_sector(tvdev, isbuf, isbufsz) != 0) {
			/* IS is out-of-date, so we need to upgrade it */
			printf("WARNING: Out-of-date Invariant sector found. Flash reprogramming won't be\n");
			printf("failsafe, continuing in 10 seconds\n");
			sleep(10);
			program_is = 1;
		} else if (!tvdev->failsafe.valid)
			/* Not in failsafe, so it's safe to silently program the IS */
			program_is = 1;
	}

	printf("Programming HCA firmware... Flash Image Size = %d\n", ibufsz);

	if (psbuf) {
		status_count = psbufsz + ibufsz + psbufsz + ibufsz;
		if (tvdev->failsafe.images[0].valid)
			status_count += tvdev->flash_sector_sz;
	} else
		status_count = ibufsz + ibufsz;

	if (program_is) {
		/* Program the Invariant Sector first */
		status_start(status_count + isbufsz + isbufsz);
		flash_write_invariant_sector(tvdev, isbuf, isbufsz);

		addr = flash_verify_image(tvdev, 0, isbuf, isbufsz);
		if (addr < isbufsz) {
			status_stop();
			printf("Flash verify of IS FAILED @ %d (%d total)\n",
				addr, isbufsz);
			ret = 1;
			goto verify_failed;
		}
	} else
		status_start(status_count);

	if (psbuf) {
		if (tvdev->failsafe.images[0].valid)
			flash_move_pps_to_sps(tvdev);

		/* Write the PPS and image */
		flash_write_pps(tvdev, tvdev->flash_sector_sz, psbuf, psbufsz,
		sector * tvdev->flash_sector_sz, ibufsz);
		flash_write_image(tvdev, sector * tvdev->flash_sector_sz, ibuf, ibufsz);

		/* Verify the PPS and image */
		addr = flash_verify_image(tvdev, tvdev->flash_sector_sz, psbuf, psbufsz);
		if (addr < psbufsz) {
			status_stop();
			printf("Flash verify of PPS FAILED @ %d (%d total)\n",
				addr, psbufsz);
			ret = 1;
			goto verify_failed;
		}

		addr = flash_verify_image(tvdev, sector * tvdev->flash_sector_sz, ibuf, ibufsz);
		if (addr < ibufsz) {
			status_stop();
			printf("Flash verify of image FAILED @ %d (%d total)\n",
				addr, ibufsz);
			ret = 1;
			goto verify_failed;
		}

		/* Finish writing the PPS */
		flash_finish_failsafe(tvdev, tvdev->flash_sector_sz, psbuf, psbufsz);
	} else {
		flash_write_image(tvdev, 0, ibuf, ibufsz);
		addr = flash_verify_image(tvdev, 0, ibuf, ibufsz);
		if (addr < ibufsz) {
			status_stop();
			printf("Flash verify of image FAILED @ %d (%d total)\n",
				addr, ibufsz);
			ret = 1;
			goto verify_failed;
		}
	}

	status_stop();

	ret = 0;

	printf("Flash verify passed!\n");

verify_failed:
	close_hca(tvdev);

	return ret;
}

static int modify_options(int hca, char *change_option_cmd)
{
	struct tvdevice *tvdev;
	union vsd *vsd;
	unsigned char *psbuf;
	int i;

	tvdev = find_device(hca);
	if (!tvdev) {
		fprintf(stderr, "couldn't find HCA #%d on the PCI bus\n", hca);
		return 1;
	}

	if (open_hca(tvdev)) {
		fprintf(stderr, "couldn't open hca %d\n", hca);
		return 1;
	}

	flash_check_failsafe(tvdev);

	psbuf = malloc(tvdev->flash_sector_sz);
	if (!psbuf) {
		fprintf(stderr, "couldn't allocate temp buffer for PPS/SPS (size = %d)\n",
			tvdev->flash_sector_sz);
		exit(1);
	}

	for (i = 0; i < tvdev->flash_sector_sz; i += 4)
		*(uint32_t *)(psbuf + i) = be32_to_cpu(flash_read(tvdev, tvdev->flash_sector_sz + i));

	vsd = (union vsd *)(psbuf + 0x20);

	if (be16_to_cpu(vsd->data.signature) != VSD_SIGNATURE_TOPSPIN ||
	    be16_to_cpu(vsd->data.vendor.topspin.signature2) != VSD_SIGNATURE_TOPSPIN) {
		/* Not out VSD format, so nothing to print */
		printf("error: VSD not in Topspin format\n");
		return 1;
	}

	fixup_topspin_vsd(&vsd->data.vendor.topspin);

	if (change_option_cmd[0] == 0) {
		if (vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_AUTOUPGRADE))
			printf("  auto_upgrade=yes\n");
		else
			printf("  auto_upgrade=no\n");

		if (vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_1))
			printf("  boot_enable_port_1=yes\n");
		else
			printf("  boot_enable_port_1=no\n");

		if (vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_ENABLE_PORT_2))
			printf("  boot_enable_port_2=yes\n");
		else
			printf("  boot_enable_port_2=no\n");

		if (!(vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_ENABLE_SCAN)))
			printf("  boot_service_scan=no\n");

		if (vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_WAIT_ON_ERROR))
			printf("  boot_wait_on_error=yes\n");
		else
			printf("  boot_wait_on_error=no\n");

		if (vsd->data.vendor.topspin.flags & cpu_to_be32(VSD_FLAG_BOOT_TRY_FOREVER))
			printf("  boot_try_forever=yes\n");
		else
			printf("  boot_try_forever=no\n");

		switch (be32_to_cpu(vsd->data.vendor.topspin.flags) & VSD_FLAG_BOOT_TYPE) {
		case VSD_FLAG_BOOT_TYPE_WELL_KNOWN:
			printf("  boot_type=well_known\n");
			break;
		case VSD_FLAG_BOOT_TYPE_SAVED:
			printf("  boot_type=saved\n");
			break;
		case VSD_FLAG_BOOT_TYPE_PXE:
			printf("  boot_type=pxe\n");
			break;
		case VSD_FLAG_BOOT_TYPE_DISABLE:
			printf("  boot_type=disable\n");
			break;
		default:
			printf("  boot_type=unknown\n");
			break;
		}

		printf("  boot_saved_port=%d\n", vsd->data.vendor.topspin.boot_port);
		printf("  boot_saved_ioc_num=%d\n", vsd->data.vendor.topspin.boot_ioc_num);
		printf("  boot_saved_dgid=%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x\n",
			vsd->data.vendor.topspin.boot_dgid[0],  vsd->data.vendor.topspin.boot_dgid[1],
			vsd->data.vendor.topspin.boot_dgid[2],  vsd->data.vendor.topspin.boot_dgid[3],
			vsd->data.vendor.topspin.boot_dgid[4],  vsd->data.vendor.topspin.boot_dgid[5],
			vsd->data.vendor.topspin.boot_dgid[6],  vsd->data.vendor.topspin.boot_dgid[7],
			vsd->data.vendor.topspin.boot_dgid[8],  vsd->data.vendor.topspin.boot_dgid[9],
			vsd->data.vendor.topspin.boot_dgid[10], vsd->data.vendor.topspin.boot_dgid[11],
			vsd->data.vendor.topspin.boot_dgid[12], vsd->data.vendor.topspin.boot_dgid[13],
			vsd->data.vendor.topspin.boot_dgid[14], vsd->data.vendor.topspin.boot_dgid[15]);
		printf("  boot_saved_service_name=%02x%02x:%02x%02x:%02x%02x:%02x%02x\n",
			vsd->data.vendor.topspin.boot_service_name[0],
			vsd->data.vendor.topspin.boot_service_name[1],
			vsd->data.vendor.topspin.boot_service_name[2],
			vsd->data.vendor.topspin.boot_service_name[3],
			vsd->data.vendor.topspin.boot_service_name[4],
			vsd->data.vendor.topspin.boot_service_name[5],
			vsd->data.vendor.topspin.boot_service_name[6],
			vsd->data.vendor.topspin.boot_service_name[7]);
		printf("  boot_pxe_secs=%d\n", vsd->data.vendor.topspin.boot_pxe_secs);
	} else {
		if (parse_change_options(&vsd->data.vendor.topspin, change_option_cmd))
			return 1;

		/* Calculate the VSD checksum */
		vsd->data.checksum = 0;
		vsd->data.checksum = cpu_to_be16(flash_crc16(vsd->raw, sizeof(vsd->raw)));

		/* Set signature and CRC to all-ones for now */
		*(uint32_t *)(psbuf + 0x8) = 0xFFFFFFFF;
		*(uint16_t *)(psbuf + 0x106) = 0xFFFF;

		flash_chip_reset(tvdev);

		status_start(tvdev->flash_sector_sz);

		status_update('E', 0);
		flash_erase_sector(tvdev, tvdev->flash_sector_sz);
		flash_write_block(tvdev, 'F', tvdev->flash_sector_sz, psbuf, tvdev->flash_sector_sz);
		status_mark();

		flash_chip_reset(tvdev);

		flash_finish_failsafe(tvdev, tvdev->flash_sector_sz, psbuf, tvdev->flash_sector_sz);

		status_stop();
	}

	free(psbuf);

	close_hca(tvdev);

	return 0;
}

static void usage(void)
{
	fprintf(stderr, "Usage: %s [OPTION]...\n", argv0);
	fprintf(stderr, "  -q\t\tProduce less output (quieter)\n");
	fprintf(stderr, "  -v\t\tProduce more output (verbose)\n");
	fprintf(stderr, "  -f\t\tForce operation (use cautiously)\n");
	fprintf(stderr, "  -p\t\tUser PCI config space access instead of memory mapping\n");
	fprintf(stderr, "  -h <num>\tSelect HCA to use for operation (default: 0)\n");
	fprintf(stderr, "  -o [<cmd>]\tChange options command (no command shows current values)\n");
	fprintf(stderr, "    The following commands and values are accepted:\n");
	fprintf(stderr, "      auto_upgrade              [yes|no]\n");
	fprintf(stderr, "      boot_enable_port1         [yes|no]\n");
	fprintf(stderr, "      boot_enable_port2         [yes|no]\n");
	fprintf(stderr, "      boot_wait_on_error        [yes|no]\n");
	fprintf(stderr, "      boot_try_forever          [yes|no]\n");
	fprintf(stderr, "      boot_type                 [well_known|saved|pxe|disable]\n");
	fprintf(stderr, "      boot_saved_port           [0|1|2]\n");
	fprintf(stderr, "      boot_saved_ioc_num        ##\n");
	fprintf(stderr, "      boot_saved_dgid           ####:####:####:####:####:####:####:####\n");
	fprintf(stderr, "      boot_saved_service_name   ####:####:####:####\n");
	fprintf(stderr, "      boot_pxe_secs             ##\n");
	fprintf(stderr, "  -u\t\tDisable autoupgrade of firmware (-o auto_upgrade=no)\n\n");

	fprintf(stderr, "One operation mode must be specified:\n");
	fprintf(stderr, "  -i\t\tIdentify HCAs\n");
	fprintf(stderr, "  -g\t\tPrint GUIDs programmed into firmware\n");
	fprintf(stderr, "  -s <filename>\n\t\tSave firmware programmed on flash device to local file\n");
	fprintf(stderr, "  [-d] <filename> [<Node GUID>]\n\t\tDownload firmware to flash device from local file\n\n");

	fprintf(stderr, "When specifying a GUID, use only the lower 24 bits in hexadecimal format.\n");

	exit(1);
}

int main(int argc, char *argv[])
{
	char *filename = NULL, *guid = NULL, *change_option_cmd = NULL;
	struct winsize winsize;
	struct pci_access *pacc;
	enum operation operation = OP_NONE;
	enum identify_mode identify_mode = IDENTIFY_EXTENDED;
	int ret, autoupgrade = 1, hca = -1;

	/* Check the size is correct. The compiler will optimize this away */
	{
		union vsd vsd;

		if (sizeof(vsd.data) != sizeof(vsd.raw)) {
			extern void __sizeof_vsd_data_doesnt_equal_sizeof_vsd_raw();

			__sizeof_vsd_data_doesnt_equal_sizeof_vsd_raw();
		}
	}

	/* Figure out the size of the window, but fall back if we can't */
	if (ioctl(0, TIOCGWINSZ, &winsize) < 0)
		cols = 80;
	else
		cols = winsize.ws_col;

	argv0 = argv[0];

	while (1) {
		int c;

		c = getopt(argc, argv, "qvfupgh:i::s:d:o::");
		if (c == -1)
			break;

		switch (c) {
		case 'q':
			verbose--;
			break;
		case 'v':
			verbose++;
			break;
		case 'f':
			force++;
			break;
		case 'u':
			autoupgrade = 0;
			break;
		case 'p':
			use_config_space = 1;
			break;
		case 'h': {
			char *p;

			ret = strtoul(optarg, &p, 0);
			if (ret == ULONG_MAX || *p) {
				fprintf(stderr, "parameter to -h option is invalid\n");
				return 1;
			}

			hca = (unsigned int)ret;
			break;
			}
		case 'i':
			if (operation != OP_NONE) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			operation = OP_IDENTIFY;
			if (optarg) {
				if (strcasecmp(optarg, "extended") == 0)
					identify_mode = IDENTIFY_EXTENDED;
				else if (strcasecmp(optarg, "pfwlabel") == 0)
					identify_mode = IDENTIFY_PRIMARY_FIRMWARE_LABEL;
				else if (strcasecmp(optarg, "hwlabel") == 0)
					identify_mode = IDENTIFY_HARDWARE_LABEL;
				else {
					fprintf(stderr, "unknown identify mode '%s'\n",
						optarg);
					usage();
				}
			} else
				identify_mode = IDENTIFY_EXTENDED;
			break;
		case 'g':
			if (operation != OP_NONE) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			operation = OP_PRINT_GUIDS;
			break;
		case 's':
			if (operation != OP_NONE) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			operation = OP_DOWNLOAD;	/* Save firmware from HCA to disk */
			filename = optarg;
			break;
		case 'd':
			if (operation != OP_NONE && operation != OP_MODIFY_OPTIONS) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			operation = OP_UPLOAD;	/* Download firmware to HCA */
			filename = optarg;
			break;
		case 'o':
			if (operation != OP_NONE && operation != OP_UPLOAD) {
				fprintf(stderr, "multiple operations specified on command line\n");
				usage();
			}

			if (optarg)
				change_option_cmd = optarg;
			else
				change_option_cmd = "";

			/* operation will be set below */
			break;
		}
	}

	if (operation == OP_NONE) {
		if (change_option_cmd) {
			operation = OP_MODIFY_OPTIONS;
		} else {
			/* Default to OP_UPLOAD, but we need a filename argument */
			if (optind >= argc)
				usage();

			operation = OP_UPLOAD;	/* Download firmware to HCA */
			filename = argv[optind++];
		}
	}

	/* The upload operation can take an optional GUID argument too */
	if (optind < argc && operation == OP_UPLOAD)
		guid = argv[optind++];

	/* Identify operation can take an optional filename argument here */
	if (optind < argc && operation == OP_IDENTIFY) {
		filename = argv[optind++];
		if (filename && identify_mode == IDENTIFY_HARDWARE_LABEL) {
			fprintf(stderr, "cannot print hardware label for firmware image file\n");
			usage();
		}
	}

	/* Change option command can take an optional filename argument here */
	if (optind < argc && change_option_cmd && change_option_cmd[0] == 0)
		change_option_cmd = argv[optind++];

	/* Set change_option_cmd correctly */
	if (!autoupgrade) {
		if (change_option_cmd && change_option_cmd[0] != 0) {
			size_t len = strlen(change_option_cmd) + strlen("auto_upgrade=off") + 1;
			char *p;

			p = malloc(len);
			if (!p) {
				fprintf(stderr, "Unable to allocate %Zd bytes for change_option_cmd\n",
					len);
				exit(1);
			}

			snprintf(p, len - 1, "%s,auto_upgrade=off",
				change_option_cmd);
			change_option_cmd = p;
		} else
			change_option_cmd = "auto_upgrade=off";
	}

	pacc = pci_alloc();
	pci_init(pacc);
	pci_scan_bus(pacc);

	switch (operation) {
	case OP_IDENTIFY:
		if (filename)
			ret = identify_firmware(filename, identify_mode);
		else {
			scan_devices(pacc);
			ret = identify_hcas(hca, identify_mode);
		}
		break;
	case OP_PRINT_GUIDS:
		scan_devices(pacc);
		ret = print_guids(hca);
		break;
	case OP_DOWNLOAD:
		scan_devices(pacc);
		ret = download_firmware(hca >= 0 ? hca : 0, filename);
		break;
	case OP_UPLOAD:
		scan_devices(pacc);
		ret = upload_firmware(hca >= 0 ? hca : 0, filename, guid, change_option_cmd);
		break;
	case OP_MODIFY_OPTIONS:
		scan_devices(pacc);
		ret = modify_options(hca >= 0 ? hca : 0, change_option_cmd);
		break;
	default:
		fprintf(stderr, "ERROR: Invalid operation %d\n", operation);
		ret = 1;
		break;
	}

	pci_cleanup(pacc);

	return ret;
}
